const recordWorker = function () {
    let self = this;
    this.onmessage = function (e) {
        switch (e.data.command) {
            case "transform":
                transform.transaction(e.data, e.data.is16K);
                break;
        }
    };

    let transform = {
        transaction(audioData, is16K) {
            let output;
            if (is16K === 16000) {
                output = transform.to16BitPCM(audioData.buffer, is16K);
            } else if (is16K > 16000) {
                output = transform.to16kHz(audioData.buffer, is16K);
                output = transform.to16BitPCM(output);
            } else if (is16K < 16000) {
                output = transform.lessTo16kHz(audioData.buffer, is16K);
                output = transform.to16BitPCM(output);
            }
            output = Array.from(new Uint8Array(output.buffer));
            self.postMessage(output);
        },
        /**
         * 大于16kHz降采样到16kHz
         * @param audioData
         * @param originalSampleRate
         * @return {*|Float32Array}
         */
        to16kHz(audioData, originalSampleRate) {
            if (originalSampleRate <= 16000) {
                return audioData;
            }

            let data = new Float32Array(audioData);
            let fitCount = Math.round(data.length * (16000 / originalSampleRate));
            let newData = new Float32Array(fitCount);
            let springFactor = (data.length - 1) / (fitCount - 1);

            newData[0] = data[0];
            for (let i = 1; i < fitCount - 1; i++) {
                let tmp = i * springFactor;
                let before = Math.floor(tmp).toFixed();
                let after = Math.ceil(tmp).toFixed();
                let atPoint = tmp - before;
                newData[i] = data[before] + (data[after] - data[before]) * atPoint;
            }
            newData[fitCount - 1] = data[data.length - 1];

            return newData;
        },
        /**
         * 小于16kHz升采样到16kHz
         * @param audioData
         * @param originalSampleRate
         * @return {*|Float32Array}
         */
        lessTo16kHz(audioData, originalSampleRate) {
            // 如果原始采样率已经是16kHz或更高，直接返回原始数据
            if (originalSampleRate >= 16000) {
                return audioData;
            }
            // 计算新的采样率与原始采样率的比例
            const upsampleFactor = 16000 / originalSampleRate;
            // 计算新的数据长度
            const newLength = Math.ceil(audioData.length * upsampleFactor);
            // 创建一个新的 Float32Array 用于存储升采样后的数据
            const newData = new Float32Array(newLength);
            // 进行最近邻插值
            for (let i = 0; i < newLength; i++) {
                // 计算原始数据中对应的样本索引
                const originalIndex = Math.floor(i / upsampleFactor);
                // 将原始样本值复制到新数据中
                newData[i] = audioData[originalIndex];
            }
            return newData;
        },
        to16BitPCM(input) {
            let dataLength = input.length * (16 / 8); // 计算了将要创建的缓冲区的长度，这里使用了位操作来表示字节数，即 16 / 8 表示每个样本占用2个字节（16位PCM格式）。
            let dataBuffer = new ArrayBuffer(dataLength); // 是一个用于存储二进制数据的固定大小的缓冲区，其长度为 dataLength 字节。
            let dataView = new DataView(dataBuffer); // 则是一个用于读写 dataBuffer 中数据的视图，通过它可以以不同的数据格式（如整数、浮点数）访问缓冲区中的数据。

            let offset = 0; // offset 变量用于追踪当前写入到 dataView 中的位置。初始为0，每次迭代增加2，因为每个16位样本占用2个字节。
            for (let i = 0; i < input.length; i++, offset += 2) {
                let s = Math.max(-1, Math.min(1, input[i])); // Math.max(-1, Math.min(1, input[i])) 确保将输入值限制在 -1 到 1 之间，因为16位PCM格式只能表示在 -32768 到 32767 之间的整数。
                // dataView.setInt16(offset, value, littleEndian) 将处理过的音频数据写入到 dataView 中：
                //     offset 是数据在缓冲区中的偏移量。
                //     value 是通过将输入数据映射到16位有符号整数的方式得到的数值。
                //     littleEndian 参数设为 true，表示使用小端序（低位字节在前，高位字节在后）存储数据，这是常见的音频格式存储方式。
                dataView.setInt16(offset, s < 0 ? s * 0x8000 : s * 0x7fff, true);
            }
            return dataView;
        },
        to8BitPCM(input) {
            var dataLength = input.length;
            var dataBuffer = new ArrayBuffer(dataLength);
            var dataView = new DataView(dataBuffer);
            var offset = 0;
            for (var i = 0; i < input.length; i++, offset++) {
                var s = Math.max(-1, Math.min(1, input[i]));
                var val = s < 0 ? s * 0x80 : s * 0x7F;
                var intVal = Math.round(val); // Round to nearest integer
                dataView.setInt8(offset, intVal);
            }
            return dataView;
        },
    };
};

export default recordWorker;
