package com.kwan.springcloudalibaba.controller;

import com.kwan.springcloudalibaba.api.FeignUserService;
import com.kwan.springcloudalibaba.common.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.cloud.client.loadbalancer.LoadBalancerClient;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;

@RefreshScope
@RestController
@RequestMapping("/nacos/consumer")
public class ConsumerUserController {
    @Autowired
    private LoadBalancerClient loadBalancerClient;
    @Autowired
    private RestTemplate restTemplate;
    @Autowired
    private FeignUserService feignUserService;
    @Autowired
    private DiscoveryClient discoveryClient;

    /**
     * 使用LoadBalanceClient和RestTemplate结合的方式来访问
     * http://127.0.0.1:8085/nacos/consumer/loadBalancerClient/1
     * 使用服务名访问
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/loadBalancerClient/{id}", produces = MediaType.APPLICATION_PROBLEM_JSON_VALUE)
    public Result loadBalancerClient(@PathVariable("id") Integer id) {
        ServiceInstance serviceInstance = loadBalancerClient.choose("nacos-server-producer");
        String url = String.format("http://%s:%s/user/%s", "nacos-server-producer", serviceInstance.getPort(), id);
        System.out.println("request url:" + url);
        return Result.ok(restTemplate.getForObject(url, Result.class));
    }

    /**
     * 使用ip访问
     * http://127.0.0.1:8085/nacos/consumer/loadBalancerClientIp/1
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/loadBalancerClientIp/{id}", produces = MediaType.APPLICATION_PROBLEM_JSON_VALUE)
    public Result loadBalancerClientIp(@PathVariable("id") Integer id) {
        ServiceInstance serviceInstance = loadBalancerClient.choose("nacos-server-producer");
        String url = String.format("http://%s:%s/user/%s", serviceInstance.getHost(), serviceInstance.getPort(), id);
        System.out.println("request url:" + url);
        //必须要用新的RestTemplate
        RestTemplate restTemplate = new RestTemplate();
        return Result.ok(restTemplate.getForObject(url, Result.class));
    }

    /**
     * 通过服务名进行负载均衡访问
     * http://127.0.0.1:8085/nacos/consumer/serverName/1
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/serverName/{id}", produces = MediaType.APPLICATION_PROBLEM_JSON_VALUE)
    public Result serverName(@PathVariable("id") Integer id) {
        String url = "http://nacos-server-producer/user/" + id;
        System.out.println("request url:" + url);
        return Result.ok(restTemplate.getForObject(url, Result.class));
    }

    /**
     * feign调用
     * http://127.0.0.1:8085/nacos/consumer/1
     * <p>
     * http://localhost:9091/user/1
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_PROBLEM_JSON_VALUE)
    public Result getUserById(@PathVariable Integer id) {
        return Result.ok(feignUserService.selectOne(id));
    }


    /**
     * 获取服务实例的具体信息
     * 如:实例个数
     * 实例url
     * 实例ip
     * 实例在nacos中的定位信息
     * http://127.0.0.1:8085/nacos/consumer/services/nacos-server-producer
     *
     * @param service
     * @return
     */
    @GetMapping("/services/{service}")
    public Result services(@PathVariable String service) {
        return Result.ok(discoveryClient.getInstances(service));
    }

    /**
     * 获取所有服务
     * http://127.0.0.1:8085/nacos/consumer/services
     *
     * @return
     */
    @GetMapping("/services")
    public Result services() {
        System.out.println(discoveryClient.description());
        System.out.println(discoveryClient.getOrder());
        return Result.ok(discoveryClient.getServices());
    }

    /**
     * 获取配置的变量
     */
    @GetMapping(value = "/nameInfo", produces = MediaType.APPLICATION_PROBLEM_JSON_VALUE)
    public Result nameInfo() {
        return Result.ok();
    }
}