package com.kwan.shuyu.netty_03_nio.c5_attach;

import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Iterator;

import static com.kwan.shuyu.until.ByteBufferUtil.debugAll;

/**
 * NioServer 服务端 NioServer 附件传递和处理消息边界
 *
 * @author : qinyingjie
 * @version : 2.2.0
 * @date : 2023/4/18 18:22
 */
@Slf4j
public class NioServer {

    private static void split(ByteBuffer source) {
        source.flip();
        final int limit = source.limit();
        for (int i = 0; i < limit; i++) {
            //找到完整的消息
            if (source.get(i) == '\n') {
                final int len = i + 1 - source.position();
                final ByteBuffer target = ByteBuffer.allocate(len);
                for (int j = 0; j < len; j++) {
                    //从source读取,向target写入
                    target.put(source.get());
                }
                debugAll(target);
            }
        }
        source.compact();
    }

    public static void main(String[] args) throws IOException {
        final Selector selector = Selector.open();
        //预设ByteBuffer,并分配空间,为了读取数据
        final ByteBuffer buffer = ByteBuffer.allocate(16);
        //创建ServerSocketChannel
        final ServerSocketChannel ssc = ServerSocketChannel.open();
        //绑定端口号
        ssc.configureBlocking(false);
        //ServerSocketChannel注册到Selector中
        final SelectionKey sscKey = ssc.register(selector, 0, null);
        sscKey.interestOps(SelectionKey.OP_ACCEPT);
        log.info("sscKey={}", sscKey);
        ssc.bind(new InetSocketAddress(8080));

        while (true) {
            //select方法,没有事件发生,线程阻塞,有事件,线程才会恢复运行
            //select在事件未处理时,它不会阻塞,会一直请求处理
            selector.select();
            final Iterator<SelectionKey> iterator = selector.selectedKeys().iterator();
            while (iterator.hasNext()) {
                final SelectionKey key = iterator.next();
                iterator.remove();
                log.info("key={}", key);
                if (key.isAcceptable()) {
                    final ServerSocketChannel channel = (ServerSocketChannel) key.channel();
                    final SocketChannel sc = channel.accept();
                    sc.configureBlocking(false);
                    ByteBuffer bf = ByteBuffer.allocate(16);
                    //将一个ByteBuffer作为附件绑定到SelectionKey,处理消息边界
                    final SelectionKey scKey = sc.register(selector, 0, bf);
                    scKey.interestOps(SelectionKey.OP_READ);
                    log.info("sc={}", sc);
                    log.info("scKey={}", scKey);
                } else if (key.isReadable()) {
                    try {
                        final SocketChannel channel = (SocketChannel) key.channel();
                        //获取附件的ByteBuffer
                        final ByteBuffer bf = (ByteBuffer) key.attachment();
                        final int read = channel.read(bf);
                        if (read == -1) {
                            key.cancel();
                        } else {
                            split(bf);//多次读取,处理消息边界
                            if (bf.position() == bf.limit()) {
                                //扩容
                                final ByteBuffer newByteBuffer = ByteBuffer.allocate(buffer.capacity() * 2);
                                buffer.flip();
                                newByteBuffer.put(buffer);
                                key.attach(newByteBuffer);//绑定新的ByteBuffer
                            }
                        }
                    } catch (IOException e) {
                        e.printStackTrace();
                        key.cancel();//取消事件处理
                    }
                }
            }
        }
    }
}