

/*
 * Copyright (c) 2019 TAOS Data, Inc. <jhtao@taosdata.com>
 *
 * This program is free software: you can use, redistribute, and/or modify
 * it under the terms of the GNU Affero General Public License, version 3
 * or later ("AGPL"), as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <iconv.h>
#include <sys/stat.h>
#include <sys/syscall.h>

#include "os.h"
#include "tscompression.h"
#include "tdataformat.h"

#define FT_CNT  8

#include <stdio.h>
#include <stdlib.h>


struct timeval startTime;
struct timeval endTime;  /* Start and end times */
struct timeval costStart; /*only used for recording the cost*/
double totalCost = 0;
int notsame_cnt  = 0;


void cost_start()
{
	  totalCost = 0;
    gettimeofday(&costStart, NULL);
}

double cost_end(const char* tag)
{
    double elapsed;
    struct timeval costEnd;
    gettimeofday(&costEnd, NULL);
    elapsed = ((costEnd.tv_sec*1000000+costEnd.tv_usec)-(costStart.tv_sec*1000000+costStart.tv_usec))/1000000.0;
    totalCost += elapsed;
    double use_ms = totalCost*1000;
    printf(" timecost %s : %.3f ms\n", tag, use_ms);
    return use_ms; 
}

float toFloat(char* buf){
  return (float)atoll(buf);
}


//
//  read float 
//
float* read_float(const char* inFile, int* pcount){
  // check valid
  if(inFile == NULL || inFile[0] == 0 ){
    printf(" inFile  is NULL or EMPTY.\n");
    return NULL;
  }

  // read in file
  FILE* pfin =  fopen(inFile, "r");
  if(pfin == NULL){
    printf(" open IN file %s error. errno=%d\n", inFile, errno);
    return false;
  }

  // read one line from infile and write to outfile
  char buf[256]={0};
  int  malloc_cnt = 100000;
  float* floats = malloc(malloc_cnt*sizeof(float));
  int  fi = 0;
  while(fgets(buf, sizeof(buf), pfin) != NULL) {
    // get item
    floats[fi] = atof(buf);
    //printf(" buff=%s float=%.50f \n ", buf, floats[fi]);
    if ( ++fi == malloc_cnt ) {
      malloc_cnt += 100000;
      floats = realloc(floats, malloc_cnt*sizeof(float));
    }
    memset(buf, 0, sizeof(buf));
  }

  // close
  fclose(pfin);
  if(pcount)
    *pcount = fi;
  return floats;
}

float check_same(float* ft1, float* ft2, int count){
  int same_cnt =0;
  for(int i=0; i< count; i++){
    
    if(ft1[i] == ft2[i]){
       same_cnt ++;
    }

    if(i < 20){
      printf(" i=%d ft1=%.50f diff=%.50f \n", i, ft1[i], ft1[i] - ft2[i]);
      printf(" i=%d ft2=%.50f \n", i, ft2[i]);
    }
      
  }
  float same_rate = same_cnt*100/count;
  printf(" all count=%d same=%d same rate=%.0f%% \n", count, same_cnt, same_rate);
  return same_rate;
}

//
// test compress and decompress
//
extern bool gOpenLossy;
bool testFile(const char* inFile, char algorithm){
  // check valid
  if(inFile == NULL || inFile[0] == 0 ){
    printf(" inFile is NULL or EMPTY.\n");
    return false;
  }

  int cnt = 0;
  float* floats = read_float(inFile, &cnt);
  if(floats == NULL) {
    return false;
  }

  // compress
  const char* input = (const char*)floats;
  int input_len = cnt * sizeof(cnt);
  char* output = (char*) malloc(input_len);
  int output_len = input_len;
  char* buff = (char*) malloc(input_len);
  int buff_len = input_len;

  printf(" file %s have count=%d \n", inFile, cnt);

  cost_start();
  int ret_len = 0;
  if(algorithm == 2) 
     ret_len = tsCompressFloat(input, input_len, cnt, output, output_len, algorithm, buff, buff_len);
  else
     ret_len = tsCompressFloatLossy(input, input_len, cnt, output, output_len, algorithm, buff, buff_len);

  if(ret_len == -1) {
    printf(" compress error.\n");
    return 0;
  }
  double use_ms1 = cost_end("compress");

  printf(" compress len=%d input len=%d\n", ret_len, input_len);
  float rate=100*(float)ret_len/(float)input_len;
  printf(" compress rate=%.1f an-rate=%.4f%%\n", (float)input_len/(float)ret_len, rate);
  
  //   
  // decompress
  //
  float* ft2 = (float*)malloc(input_len); 
  cost_start();
  int code = 0;

  if(algorithm == 2) 
     code = tsDecompressFloat(output, ret_len, cnt, (char*)ft2, input_len, algorithm, buff, buff_len);
  else
     code = tsDecompressFloatLossy(output, ret_len, cnt, (char*)ft2, input_len, algorithm, buff, buff_len);
   

  double use_ms2 = cost_end("Decompress");
  printf(" Decompress return length=%d \n", code);

  // compare same
  float same_rate = check_same(floats, ft2, cnt);

  printf("\n ------------------  count:%d TD <%s> ---------------- \n", cnt, algorithm == 2?"TD":"SZ");
  printf("    Compress Rate ......... [%.0f%%] \n", rate);
  double speed1 = (cnt*sizeof(float)*1000/1024/1024)/use_ms1;
  printf("    Compress Time ......... [%.4fms] speed=%.1f MB/s\n", use_ms1, speed1);
  double speed2 = (cnt*sizeof(float)*1000/1024/1024)/use_ms2;
  printf("    Decompress Time........ [%.4fms] speed=%.1f MB/s\n", use_ms2, speed2);
  printf("    Same Rate ............. [%.0f%%] \n\n", same_rate);


  // free
  free(ft2);
  free(floats);
  free(buff);
  free(output); 

  return true;
}
//
//  txt to binary file
//
#define BUFF_CNT 256
bool txt_to_bin(const char* inFile, const char* outFile){
  // check valid
  if(inFile == NULL || outFile == NULL || inFile[0] == 0 || outFile[0] == 0){
    printf(" inFile or outFile is NULL or EMPTY.\n");
    return false;
  }

  printf(" infile=%s \n",  inFile);
  printf(" outfile=%s \n", outFile);

  // read in file
  FILE* pfin =  fopen(inFile, "r");
  if(pfin == NULL){
    printf(" open IN file %s error. errno=%d\n", inFile, errno);
    return false;
  }

  // create out file
  FILE* pfout = fopen(outFile, "w+");
  if(pfout == NULL){
    printf(" open OUT file %s error. errno=%d\n", outFile, errno);
    fclose(pfin);
    return false;
  }

  // read one line from infile and write to outfile
  char buf[256]={0};
  float fbuf[BUFF_CNT] = {0};
  int   fi = 0;
  int   count = 0;
  while(fgets(buf, sizeof(buf), pfin) != NULL) {
    // get item
    fbuf[fi] = toFloat(buf);
    if ( ++fi == BUFF_CNT ) {
      // write
      if(fwrite(fbuf, sizeof(float), BUFF_CNT, pfout) == 0) {
        printf(" write to file %s error , code=%d . app exit.\n", outFile, errno);
        exit(1);
      }
      fi = 0;
    }
    count ++;
    memset(buf, 0, sizeof(buf));
  }

  // write retain
  if( fi > 0){
    if(fwrite(fbuf, sizeof(float), fi, pfout) == 0) {
      printf(" write to file %s error , code=%d . app exit.\n", outFile, errno);
      exit(1);
    }
  }

  // close
  fclose(pfin);
  fclose(pfout);

  // total
  printf(" count=%d  write bytes=%lu \n", count, count*sizeof(float));
  return true;
}


int memTestDouble() {
  //
  //  compress
  //
  //float ft1[] = {1.2, 2.4, 3.33, 4.444, 5.555, 6.6666, 7.7777, 8.88888,1.2, 2.4, 3.33, 4.444, 5.555, 6.6666, 7.7777, 8.88888,1.2, 2.4, 3.33, 4.444, 5.555, 6.6666, 7.7777, 8.88888};
  ///*
  double ft1[ ] = {\
      3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.9010000228881836, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.630000114440918, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7079999446868896, 3.7079999446868896, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7119998931884766, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7100000381469727, 3.7079999446868896, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7070000171661377, 3.7100000381469727, 3.7119998931884766, 3.7119998931884766, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7139999866485596, 3.7109999656677246, 3.7109999656677246, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7130000591278076, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7119998931884766, 3.7130000591278076, 3.7109999656677246, 3.7079999446868896, 3.7090001106262207, 3.7090001106262207, 3.7119998931884766, 3.7109999656677246, 3.7109999656677246, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7090001106262207, 3.7109999656677246, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7079999446868896, 3.7079999446868896, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7090001106262207, 3.7100000381469727, 3.7090001106262207, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7109999656677246, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7149999141693115, 3.7079999446868896, 3.7070000171661377, 3.7049999237060547, 3.7060000896453857, 3.7060000896453857\
     ,3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.634000062942505, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422
     ,3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.9010000228881836, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.630000114440918, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7079999446868896, 3.7079999446868896, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7119998931884766, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7100000381469727, 3.7079999446868896, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7070000171661377, 3.7100000381469727, 3.7119998931884766, 3.7119998931884766, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7139999866485596, 3.7109999656677246, 3.7109999656677246, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7130000591278076, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7119998931884766, 3.7130000591278076, 3.7109999656677246, 3.7079999446868896, 3.7090001106262207, 3.7090001106262207, 3.7119998931884766, 3.7109999656677246, 3.7109999656677246, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7090001106262207, 3.7109999656677246, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7079999446868896, 3.7079999446868896, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7090001106262207, 3.7100000381469727, 3.7090001106262207, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7109999656677246, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7149999141693115, 3.7079999446868896, 3.7070000171661377, 3.7049999237060547, 3.7060000896453857, 3.7060000896453857\
     ,3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.634000062942505, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422
     };
//*/
  const char* input = (const char*) ft1;
  int input_len = sizeof(ft1);
  int cnt = input_len/sizeof(double);
  printf(" input input_len=%d count=%d \n", input_len, cnt);

  
  char* output = (char*) malloc(input_len);
  int output_len = input_len;

  char buff[1024] ={0};
  int buff_len = sizeof(buff);

  cost_start();
  int ret_len = tsCompressDoubleLossy(input, input_len, cnt, output, output_len, ONE_STAGE_COMP, buff, buff_len);
  if(ret_len == -1) {
    printf(" compress error.\n");
    return 0;
  }
  cost_end(" tscompress");

  printf(" compress return len=%d input len=%d\n", ret_len, input_len);
  printf(" compress rate=%.1f an-rate=%.0f%%\n", (float)input_len/(float)ret_len, 100*(float)ret_len/(float)input_len);
  
  //   
  // decompress
  //
  double* ft2 = (double*)malloc(input_len); 
  cost_start();
  int code = tsDecompressDoubleLossy(output, ret_len, cnt, (char*)ft2, input_len, ONE_STAGE_COMP, buff, buff_len);
  cost_end("tsde-compress double");
  printf(" de-compress return code=%d \n", code);


  //
  // show
  //
  int  same_cnt = 0;
  int  diffshow_cnt = 0;
  for(int i=0; i< cnt; i++){

    if(i < 10)
      printf(" i=%d ft1=%.20f ft2=%.20f  same=%d\n", i, ft1[i], ft2[i], ft1[i] == ft2[i]);

    if(ft1[i] == ft2[i])
        same_cnt ++;
    else {
      notsame_cnt++;
      if(i >= 10 && ++diffshow_cnt < 50){
         printf(" i=%d ft1=%.20f ft2=%.20f  diff\n", i, ft1[i], ft2[i]);
      }
    }
  }
   
  printf(" ---- result doulbe : same =%d cnt=%d  rate:%d%% global not same=%d---- \n", same_cnt, cnt, same_cnt*100/cnt, notsame_cnt);
   
  free(output); 
  free(ft2);
  return 1;
}




int memTest() {
  //
  //  compress
  //
  //float ft1[] = {1.2, 2.4, 3.33, 4.444, 5.555, 6.6666, 7.7777, 8.88888,1.2, 2.4, 3.33, 4.444, 5.555, 6.6666, 7.7777, 8.88888,1.2, 2.4, 3.33, 4.444, 5.555, 6.6666, 7.7777, 8.88888};
  ///*
  float ft1[ ] = {\
      3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.9010000228881836, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.630000114440918, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7079999446868896, 3.7079999446868896, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7119998931884766, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7100000381469727, 3.7079999446868896, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7070000171661377, 3.7100000381469727, 3.7119998931884766, 3.7119998931884766, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7139999866485596, 3.7109999656677246, 3.7109999656677246, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7130000591278076, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7119998931884766, 3.7130000591278076, 3.7109999656677246, 3.7079999446868896, 3.7090001106262207, 3.7090001106262207, 3.7119998931884766, 3.7109999656677246, 3.7109999656677246, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7090001106262207, 3.7109999656677246, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7079999446868896, 3.7079999446868896, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7090001106262207, 3.7100000381469727, 3.7090001106262207, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7109999656677246, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7149999141693115, 3.7079999446868896, 3.7070000171661377, 3.7049999237060547, 3.7060000896453857, 3.7060000896453857\
     ,3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.634000062942505, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422
     ,3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8980000019073486, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8980000019073486, 3.8949999809265137, 3.8959999084472656, 3.8980000019073486, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.9010000228881836, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8980000019073486, 3.8970000743865967, 3.8989999294281006, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8949999809265137, 3.8970000743865967, 3.8980000019073486, 3.8980000019073486, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8970000743865967, 3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8959999084472656, 3.8970000743865967, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.9000000953674316, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8959999084472656, 3.8980000019073486, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.630000114440918, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.8940000534057617, 3.8959999084472656, 3.8970000743865967, 3.8970000743865967, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8929998874664307, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8959999084472656, 3.8929998874664307, 3.8949999809265137, 3.8929998874664307, 3.8959999084472656, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8940000534057617, 3.8970000743865967, 3.8940000534057617, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8949999809265137, 3.8940000534057617, 3.8940000534057617, 3.8959999084472656, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8959999084472656, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787, 3.8929998874664307, 3.8940000534057617, 3.8929998874664307, 3.8940000534057617, 3.8949999809265137, 3.8929998874664307, 3.8929998874664307, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8929998874664307, 3.8940000534057617, 3.8989999294281006, 3.8929998874664307, 3.8929998874664307, 3.8929998874664307, 3.8919999599456787, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8949999809265137, 3.8970000743865967, 3.8940000534057617, 3.8940000534057617, 3.8940000534057617, 3.8929998874664307, 3.8919999599456787\
     ,3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7079999446868896, 3.7079999446868896, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7119998931884766, 3.7090001106262207, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7100000381469727, 3.7079999446868896, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7070000171661377, 3.7100000381469727, 3.7119998931884766, 3.7119998931884766, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7139999866485596, 3.7109999656677246, 3.7109999656677246, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7130000591278076, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7119998931884766, 3.7130000591278076, 3.7109999656677246, 3.7079999446868896, 3.7090001106262207, 3.7090001106262207, 3.7119998931884766, 3.7109999656677246, 3.7109999656677246, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7090001106262207, 3.7109999656677246, 3.7119998931884766, 3.7109999656677246, 3.7090001106262207, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7109999656677246, 3.7119998931884766, 3.7079999446868896, 3.7079999446868896, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7100000381469727, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7100000381469727, 3.7070000171661377, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7090001106262207, 3.7100000381469727, 3.7090001106262207, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7079999446868896, 3.7090001106262207, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7070000171661377, 3.7090001106262207, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7070000171661377, 3.7060000896453857, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7060000896453857, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7060000896453857, 3.7060000896453857, 3.7109999656677246, 3.7070000171661377, 3.7060000896453857, 3.7060000896453857, 3.7049999237060547, 3.7070000171661377, 3.7100000381469727, 3.7079999446868896, 3.7070000171661377, 3.7070000171661377, 3.7079999446868896, 3.7079999446868896, 3.7149999141693115, 3.7079999446868896, 3.7070000171661377, 3.7049999237060547, 3.7060000896453857, 3.7060000896453857\
     ,3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.634000062942505, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.63100004196167, 3.63100004196167, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.63100004196167, 3.63100004196167, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.63100004196167, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.631999969482422, 3.632999897003174, 3.631999969482422, 3.632999897003174, 3.631999969482422
     };
//*/
  const char* input = (const char*) ft1;
  int input_len = sizeof(ft1);
  int cnt = input_len/sizeof(float);
  printf(" input input_len=%d count=%d \n", input_len, cnt);

  
  char* output = (char*) malloc(input_len);
  int output_len = input_len;

  char buff[1024] ={0};
  int buff_len = sizeof(buff);

  cost_start();
  int ret_len = tsCompressFloatLossy(input, input_len, cnt, output, output_len, ONE_STAGE_COMP, buff, buff_len);
  if(ret_len == -1) {
    printf(" compress error.\n");
    return 0;
  }
  cost_end(" tscompress");

  printf(" compress return len=%d input len=%d\n", ret_len, input_len);
  printf(" compress sz rate=%.1f an-rate=%.2f%%\n", (float)input_len/(float)ret_len, 100*(float)ret_len/(float)input_len);
  
  //   
  // decompress
  //
  float* ft2 = (float*)malloc(input_len); 
  cost_start();
  int code = tsDecompressFloatLossy(output, ret_len, cnt, (char*)ft2, input_len, ONE_STAGE_COMP, buff, buff_len);
  cost_end("tsde-compress");
  printf(" de-compress return code=%d \n", code);


  //
  // show
  //
  int  same_cnt = 0;
  int  diffshow_cnt = 0;
  for(int i=0; i< cnt; i++){

    if(i < 10)
      printf(" i=%d ft1=%.20f ft2=%.20f  same=%d\n", i, ft1[i], ft2[i], ft1[i] == ft2[i]);

    if(ft1[i] == ft2[i])
        same_cnt ++;
    else {
      notsame_cnt++;
      if(i >= 10 && ++diffshow_cnt < 50){
         printf(" i=%d ft1=%.20f ft2=%.20f  diff\n", i, ft1[i], ft2[i]);
      }
    }
  }
   
  printf(" ---- result : same =%d cnt=%d  rate:%d%% global not same=%d---- \n", same_cnt, cnt, same_cnt*100/cnt, notsame_cnt);
   
  free(output); 
  free(ft2);
  return 1;
}

void* memTestThread(void* lparam) {
  //memTest();
  printf(" enter thread ....\n");
  for(int i=0; i< 5; i++)
  {
      memTest();
      printf(" start i=%d .... \n", i);
  }
  return NULL;
}

void test_threadsafe(int thread_count){
  printf(" test thread safe . thread count=%d \n", thread_count);
  pthread_t handle[1000000];
  int i=0;
  for(i=0; i< thread_count; i++){
    printf("  create thread %d... \n", i);
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_create(&handle[i], &attr, memTestThread, NULL);
    pthread_attr_destroy(&attr);
  }

  for(i=0; i< thread_count; i++)
  {
    pthread_join(handle[i], NULL);
  }
  
  printf(" test thread safe end. not same count=%d\n", notsame_cnt);
  
}


void* memTestThreadDouble(void* lparam) {
  //memTest();
  printf(" enter thread ....\n");
  for(int i=0; i< 10; i++)
  {
      memTest();
      printf(" double start i=%d .... \n", i);
  }
  return NULL;
}

void test_threadsafe_double(int thread_count){
  printf(" test thread safe . thread count=%d \n", thread_count);
  pthread_t handle[1000000];
  int i=0;
  for(i=0; i< thread_count; i++){
    printf("  create thread %d... \n", i);
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_create(&handle[i], &attr, memTestThreadDouble, NULL);
    pthread_attr_destroy(&attr);
  }

  for(i=0; i< thread_count; i++)
  {
    pthread_join(handle[i], NULL);
  }
  
  printf("\n  ---- double test thread safe end. not same count=%d-----\n", notsame_cnt);
  
}


void unitTestFloat() {

  float ft1 [] = {1.11, 2.22, 3.333};
  int cnt = sizeof(ft1)/sizeof(float);
  float* floats = ft1;
  int algorithm = 2;

  // compress
  const char* input = (const char*)floats;
  int input_len = cnt * sizeof(float);
  int output_len = input_len + 1024;
  char* output = (char*) malloc(output_len);
  char* buff = (char*) malloc(input_len);
  int buff_len = input_len;

  printf(" ft1 have count=%d \n", cnt);
  strcpy(output, "abcde");

  cost_start();
  int ret_len = 0;
  ret_len = tsCompressFloatLossy(input, input_len, cnt, output, output_len, algorithm, buff, buff_len);

  if(ret_len == 0) {
    printf(" compress error.\n");
    return ;
  }
  double use_ms1 = cost_end("compress");

  printf(" compress len=%d input len=%d\n", ret_len, input_len);
  float rate=100*(float)ret_len/(float)input_len;
  printf(" compress rate=%.1f an-rate=%.4f%%\n", (float)input_len/(float)ret_len, rate);
  
  //   
  // decompress
  //
  float* ft2 = (float*)malloc(input_len); 
  cost_start();
  int code = 0;
  code = tsDecompressFloatLossy(output, ret_len, cnt, (char*)ft2, input_len, algorithm, buff, buff_len);
   

  double use_ms2 = cost_end("Decompress");
  printf(" Decompress return length=%d \n", code);

  // compare same
  float same_rate = check_same(floats, ft2, cnt);

  printf("\n ------------------  count:%d TD <SZ> ---------------- \n", cnt);
  printf("    Compress Rate ......... [%.0f%%] \n", rate);
  double speed1 = (cnt*sizeof(float)*1000/1024/1024)/use_ms1;
  printf("    Compress Time ......... [%.4fms] speed=%.1f MB/s\n", use_ms1, speed1);
  double speed2 = (cnt*sizeof(float)*1000/1024/1024)/use_ms2;
  printf("    Decompress Time........ [%.4fms] speed=%.1f MB/s\n", use_ms2, speed2);
  printf("    Same Rate ............. [%.0f%%] \n\n", same_rate);


  // free
  free(ft2);
  free(buff);
  free(output); 
 
}



//
//   -----------------  main ----------------------
//
int main(int argc, char *argv[]) {
  printf("welcome to use taospack tools v1.2 \n");
 
  gOpenLossy = false;
  tsLossyInit();
  //
  //tsCompressExit();
  //return 1; 

  if(argc == 3){
    char algo = 0;
    // t
    if(strcmp(argv[1], "-tone") == 0 || strcmp(argv[1], "-t") == 0 ) {
        algo = ONE_STAGE_COMP;
        gOpenLossy = true;
    }
    if(strcmp(argv[1], "-ttwo") == 0) {
        algo = TWO_STAGE_COMP;
        gOpenLossy = false;
    }

    if(strcmp(argv[1], "-sf") == 0) {
        test_threadsafe(atoi(argv[2]));
        return 0;
    }

    if(strcmp(argv[1], "-sd") == 0) {
        test_threadsafe_double(atoi(argv[2]));
        return 0;
    }
    if(algo == 0){
      printf(" no param -tone -ttwo \n");
      return 0;
    }
     
    bool ret = testFile(argv[2], algo);
    printf(" test file %s. \n", ret ? "ok" : "err");
    return 1;

  } else if( argc == 2) {
    if(strcmp(argv[1], "-mem") == 0) {
      memTest();
    } 
  }
  else{
    unitTestFloat();
  }

  //memTest();
  return 0;
}

