'''
This code is rewritten by Paddle based on
https://github.com/jina-ai/discoart/blob/main/discoart/nn/sec_diff.py
'''
import math
from dataclasses import dataclass
from functools import partial

import paddle
import paddle.nn as nn


@dataclass
class DiffusionOutput:
    v: paddle.Tensor
    pred: paddle.Tensor
    eps: paddle.Tensor


class SkipBlock(nn.Layer):

    def __init__(self, main, skip=None):
        super().__init__()
        self.main = nn.Sequential(*main)
        self.skip = skip if skip else nn.Identity()

    def forward(self, input):
        return paddle.concat([self.main(input), self.skip(input)], axis=1)


def append_dims(x, n):
    return x[(Ellipsis, *(None, ) * (n - x.ndim))]


def expand_to_planes(x, shape):
    return paddle.tile(append_dims(x, len(shape)), [1, 1, *shape[2:]])


def alpha_sigma_to_t(alpha, sigma):
    return paddle.atan2(sigma, alpha) * 2 / math.pi


def t_to_alpha_sigma(t):
    return paddle.cos(t * math.pi / 2), paddle.sin(t * math.pi / 2)


class SecondaryDiffusionImageNet2(nn.Layer):

    def __init__(self):
        super().__init__()
        c = 64  # The base channel count
        cs = [c, c * 2, c * 2, c * 4, c * 4, c * 8]

        self.timestep_embed = FourierFeatures(1, 16)
        self.down = nn.AvgPool2D(2)
        self.up = nn.Upsample(scale_factor=2, mode='bilinear', align_corners=False)

        self.net = nn.Sequential(
            ConvBlock(3 + 16, cs[0]),
            ConvBlock(cs[0], cs[0]),
            SkipBlock([
                self.down,
                ConvBlock(cs[0], cs[1]),
                ConvBlock(cs[1], cs[1]),
                SkipBlock([
                    self.down,
                    ConvBlock(cs[1], cs[2]),
                    ConvBlock(cs[2], cs[2]),
                    SkipBlock([
                        self.down,
                        ConvBlock(cs[2], cs[3]),
                        ConvBlock(cs[3], cs[3]),
                        SkipBlock([
                            self.down,
                            ConvBlock(cs[3], cs[4]),
                            ConvBlock(cs[4], cs[4]),
                            SkipBlock([
                                self.down,
                                ConvBlock(cs[4], cs[5]),
                                ConvBlock(cs[5], cs[5]),
                                ConvBlock(cs[5], cs[5]),
                                ConvBlock(cs[5], cs[4]),
                                self.up,
                            ]),
                            ConvBlock(cs[4] * 2, cs[4]),
                            ConvBlock(cs[4], cs[3]),
                            self.up,
                        ]),
                        ConvBlock(cs[3] * 2, cs[3]),
                        ConvBlock(cs[3], cs[2]),
                        self.up,
                    ]),
                    ConvBlock(cs[2] * 2, cs[2]),
                    ConvBlock(cs[2], cs[1]),
                    self.up,
                ]),
                ConvBlock(cs[1] * 2, cs[1]),
                ConvBlock(cs[1], cs[0]),
                self.up,
            ]),
            ConvBlock(cs[0] * 2, cs[0]),
            nn.Conv2D(cs[0], 3, 3, padding=1),
        )

    def forward(self, input, t):
        timestep_embed = expand_to_planes(self.timestep_embed(t[:, None]), input.shape)
        v = self.net(paddle.concat([input, timestep_embed], axis=1))
        alphas, sigmas = map(partial(append_dims, n=v.ndim), t_to_alpha_sigma(t))
        pred = input * alphas - v * sigmas
        eps = input * sigmas + v * alphas
        return DiffusionOutput(v, pred, eps)


class FourierFeatures(nn.Layer):

    def __init__(self, in_features, out_features, std=1.0):
        super().__init__()
        assert out_features % 2 == 0
        # self.weight = nn.Parameter(paddle.randn([out_features // 2, in_features]) * std)
        self.weight = paddle.create_parameter([out_features // 2, in_features],
                                              dtype='float32',
                                              default_initializer=nn.initializer.Normal(mean=0.0, std=std))

    def forward(self, input):
        f = 2 * math.pi * input @ self.weight.T
        return paddle.concat([f.cos(), f.sin()], axis=-1)


class ConvBlock(nn.Sequential):

    def __init__(self, c_in, c_out):
        super().__init__(
            nn.Conv2D(c_in, c_out, 3, padding=1),
            nn.ReLU(),
        )
