/* Copyright (c) 2020 PaddlePaddle Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. */
#include <algorithm>
#include <memory>
#include <string>
#include <unordered_map>
#include <vector>

#include "paddle/fluid/framework/no_need_buffer_vars_inference.h"
#include "paddle/fluid/framework/var_type_inference.h"
#include "paddle/fluid/framework/op_registry.h"

#include "mpc_lookup_table_v2_op.h"

namespace paddle {
namespace operators {

class MpcLookupTableV2Op : public framework::OperatorWithKernel {
public:
    using framework::OperatorWithKernel::OperatorWithKernel;

    void InferShape(framework::InferShapeContext* ctx) const override {
        PADDLE_ENFORCE_EQ(ctx->HasInput("W"), true,
                          "Input(W) of LookupTableV2Op should not be null.");
        PADDLE_ENFORCE_EQ(ctx->HasInput("Ids"), true,
                          "Input(Ids) of LookupTableV2Op should not be null.");
        PADDLE_ENFORCE_EQ(ctx->HasOutput("Out"), true,
                          "Output(Out) of LookupTableV2Op should not be null.");

        auto table_dims = ctx->GetInputDim("W");
        auto ids_dims = ctx->GetInputDim("Ids");
        int ids_rank = ids_dims.size();
        VLOG(5) << "ids rank is " << ids_rank << std::endl;
        PADDLE_ENFORCE_EQ(
            table_dims.size(), 3,
            "ShapeError: The dimensions of the 'mpc lookup table' must be 3. "
            "But received lookup table's dimensions = %d, "
            "lookup table's shape = [%s].",
            table_dims.size(), table_dims);
        PADDLE_ENFORCE_EQ(
            ids_dims.size(), 3,
            "ShapeError: The dimensions of the 'idexes' must be 3, "
            "Other dimensions are not supported temporarily. "
            "Received idexes' dimensions = %d, "
            "idexes's shape = [%s].",
            table_dims.size(), table_dims);
        PADDLE_ENFORCE_EQ(
            table_dims[0], 2,
            "ShapeError: The first dimensions of the 'mpc lookup table' must be 2. "
            "But received lookup table's first dimensions = %d.",
            table_dims[0]);
        PADDLE_ENFORCE_EQ(
            ids_dims[0], 2,
            "ShapeError: The first dimensions of the 'indexes' must be 2. "
            "But received indexes' first dimensions = %d.",
            ids_dims[0]);

        auto output_dims = framework::vectorize(ids_dims);
        output_dims[output_dims.size() - 1] = table_dims[2];
        auto out_dims = framework::make_ddim(output_dims);
        ctx->SetOutputDim("Out", out_dims);

        if (ctx->GetOutputsVarType("Out")[0] ==
                framework::proto::VarType::LOD_TENSOR) {
            ctx->ShareLoD("Ids", /*->*/ "Out");
        }
    }

protected:
    framework::OpKernelType GetExpectedKernelType(
        const framework::ExecutionContext& ctx) const override {
        auto data_type = OperatorWithKernel::IndicateVarDataType(ctx, "W");
        return framework::OpKernelType(data_type, ctx.device_context());
    }
};

class MpcLookupTableV2OpMaker : public framework::OpProtoAndCheckerMaker {
public:
    void Make() override {
        AddInput("W",
                 "(Tensor) The input represents embedding tensors, "
                 "which is a learnable parameter.");
        AddInput("Ids",
                 "An input with type int64 "
                 "contains the ids to be looked up in W.");
        AddOutput("Out", "The lookup results, which have the same type as W.");
        AddAttr<bool>("is_sparse",
                      "(boolean, default false) "
                      "Sparse update.")
        .SetDefault(false);
        AddAttr<bool>("is_distributed",
                      "(boolean, default false) distributed lookup table.")
        .SetDefault(false);
        AddAttr<int64_t>("padding_idx",
                         "(int64, default -1) "
                         "If the value is -1, it makes no effect to lookup. "
                         "Otherwise the given value indicates padding the output "
                         "with zeros whenever lookup encounters it in Ids.")
        .SetDefault(kNoPadding);

        // for parameter prefetch
        AddAttr<bool>("remote_prefetch", "").SetDefault(false);
        AddAttr<int>("trainer_id", "trainer id from 0 ~ worker_num.").SetDefault(0);
        AddAttr<std::vector<int64_t>>("height_sections",
                                      "Height for each output SelectedRows.")
                                   .SetDefault(std::vector<int64_t>({}));
        AddAttr<std::vector<std::string>>(
                                           "epmap",
                                           "(string vector, default 127.0.0.1:6164)"
                                           "Server endpoints in the order of input variables for mapping")
                                       .SetDefault({});
        AddAttr<std::vector<std::string>>(
                                           "table_names",
                                           "(string vector, the splited table names that will be fetched from "
                                           "parameter server)"
                                           "in the order of input variables for mapping")
                                       .SetDefault({});

        AddComment(R"DOC(
Lookup Table V2 Operator.

This operator is used to perform lookups on the parameter W,
then concatenated into a dense tensor.

The input Ids can carry the LoD (Level of Details) information,
or not. And the output only shares the LoD information with input Ids.

)DOC");
  }
};

DECLARE_NO_NEED_BUFFER_VARS_INFERER(MpcLookupTableV2GradOpNoBuffer, "W");

template <typename T>
class MpcLookupTableV2GradOpMaker : public framework::SingleGradOpMaker<T> {
 public:
  using framework::SingleGradOpMaker<T>::SingleGradOpMaker;

  protected:
  void Apply(GradOpPtr<T> op) const override {
    op->SetType("mpc_lookup_table_v2_grad");

    op->SetInput("W", this->Input("W"));
    op->SetInput("Ids", this->Input("Ids"));
    op->SetInput(framework::GradVarName("Out"), this->OutputGrad("Out"));

    op->SetOutput(framework::GradVarName("W"), this->InputGrad("W"));

    op->SetAttrMap(this->Attrs());
  }
};

class MpcLookupTableV2OpGrad : public framework::OperatorWithKernel {
 public:
  using framework::OperatorWithKernel::OperatorWithKernel;

  void InferShape(framework::InferShapeContext* ctx) const override {
    auto table_dims = ctx->GetInputDim("W");
    ctx->SetOutputDim(framework::GradVarName("W"), table_dims);
  }

 protected:
  framework::OpKernelType GetExpectedKernelType(
      const framework::ExecutionContext& ctx) const override {
    auto data_type = OperatorWithKernel::IndicateVarDataType(
        ctx, framework::GradVarName("Out"));
    return framework::OpKernelType(data_type, ctx.device_context());
  }
};

class MpcLookupTableV2OpGradVarTypeInference : public framework::VarTypeInference {
 public:
  void operator()(framework::InferVarTypeContext* ctx) const override {
    auto out_var_name = framework::GradVarName("W");
    auto attr = ctx->GetAttr("is_sparse");
    bool is_sparse = boost::get<bool>(attr);
    if (is_sparse) {
      VLOG(3) << "mpc_lookup_table_v2_grad op " << framework::GradVarName("W")
              << " is set to SelectedRows";
      ctx->SetOutputType(out_var_name,
                         framework::proto::VarType::SELECTED_ROWS);
    } else {
      VLOG(3) << "mpc_lookup_table_v2_grad op " << framework::GradVarName("W")
              << " is set to LoDTensor";
      ctx->SetOutputType(out_var_name, framework::proto::VarType::LOD_TENSOR);
    }
    ctx->SetOutputDataType(out_var_name, ctx->GetInputDataType("W"));
  }
};

}  // namespace operators
}  // namespace paddle

namespace ops = paddle::operators;
REGISTER_OPERATOR(mpc_lookup_table_v2, ops::MpcLookupTableV2Op,
                  ops::MpcLookupTableV2OpMaker,
                  ops::MpcLookupTableV2GradOpMaker<paddle::framework::OpDesc>,
                  ops::MpcLookupTableV2GradOpMaker<paddle::imperative::OpBase>);

REGISTER_OPERATOR(mpc_lookup_table_v2_grad, ops::MpcLookupTableV2OpGrad,
                  ops::MpcLookupTableV2GradOpNoBuffer,
                  ops::MpcLookupTableV2OpGradVarTypeInference);

REGISTER_OP_CPU_KERNEL(mpc_lookup_table_v2, ops::MpcLookupTableV2Kernel<int64_t>);
REGISTER_OP_CPU_KERNEL(mpc_lookup_table_v2_grad,
                       ops::MpcLookupTableV2GradKernel<int64_t>);

