# PaddleLite使用X86预测部署

## 一、Docker或者Linux环境

Paddle-Lite 支持在Docker或Linux环境编译x86预测库。环境搭建参考[编译环境准备](../source_compile/compile_env)。

如果编译python库(`build_python=ON`)需要额外安装`python-dev`和`patchelf`

```shell
apt-get install patchelf
apt-get install python && apt-get install python-dev
```

### 编译

1、 下载代码

```bash
# 下载Paddle-Lite源码
git clone https://github.com/PaddlePaddle/Paddle-Lite.git
# 切换到release分支
git checkout release/v2.6.0
```

2、 源码编译

```bash
cd Paddle-Lite
# 请在Paddle-Lite当前目录下执行脚本
./lite/tools/build.sh --build_python=ON x86

# 其他可选择编译选项
# --with_log=OFF 关闭LOG信息输出
# --build_python=OFF 编译python预测库
```

### 编译结果说明

x86编译结果位于 `build.lite.x86/inference_lite_lib`
**具体内容**说明：

1、 `bin`文件夹：可执行工具文件 `test_model_bin`

2、 `cxx`文件夹：包含c++的库文件与相应的头文件

- `include`  : 头文件
- `lib` : 库文件
  - 静态库文件：
    - `libpaddle_api_full_bundled.a`  ：full_api 静态库
    - `libpaddle_api_light_bundled.a` ：light_api 静态库
  - 动态库文件：
    - `libpaddle_full_api_shared.so` ：full_api 动态库
    - `libpaddle_light_api_shared.so`：light_api 动态库

3、 `third_party` 文件夹：依赖的第三方预测库mklml

- mklml : Paddle-Lite预测库依赖的mklml数学库

4、 `demo/cxx`文件夹：x86预测库的C++ 示例demo

- `mobilenetv1_full` ：使用full_api 执行mobilenet_v1预测的C++ demo
- `mobilenetv1_light` ：使用light_api 执行mobilenet_v1预测的C++ demo

5、 `demo/python`文件夹：x86预测库的Python 示例demo

- `mobilenetv1_full_api.py` ：使用full_api 执行mobilenet_v1预测的Python demo
- `mobilenetv1_light_api.py` ：使用light_api 执行mobilenet_v1预测的Python demo

6、 `python`文件夹：包含python的库文件和对应的.whl包

- `install`文件夹：编译成功的.whl包位于`install/dist/*.whl`
- `lib`文件夹：.whl包依赖的库文件

**(若不需要编译python预测库，则将编译命令替换为`./lite/tools/build.sh x86`)**

### x86预测API使用示例

1、`mobilenetv1_full`目录结构

```bash
mobilenetv1_full/
|-- CMakeLists.txt
|-- build.sh
|-- build.bat
-- mobilenet_full_api.cc
```

本demo使用cmake构建`CMakeLists.txt`为cmake脚本，`mobilenet_full_api.cc`是x86示例的源代码、`build.sh`为编译的脚本。

2、demo使用方法

``` bash
# 1、编译
cd mobilenetv1_full
sh build.sh
```
编译结果为当前目录下的 `mobilenet_full_api `
``` bash
# 2、执行预测
./mobilenet_full_api ./mobilenet_v1
```
下载并解压模型[`mobilenet_v1`](http://paddle-inference-dist.bj.bcebos.com/mobilenet_v1.tar.gz)到当前目录，执行以上命令进行预测。

```bash
# 3、执行demo后输出结果如下，全一输入下mobilenet_v1的预测结果
Output shape 1000
Output[0]: 0.000191312
Output[100]: 0.000159713
Output[200]: 0.000264313
Output[300]: 0.000210793
Output[400]: 0.00103236
Output[500]: 0.000110071
Output[600]: 0.00482924
Output[700]: 0.00184533
Output[800]: 0.000202116
Output[900]: 0.000585591
```



3、示例源码`mobilenet_full_api.cc`

```c++
#include <iostream>
#include <vector>
#include "paddle_api.h"


using namespace paddle::lite_api;  // NOLINT

int64_t ShapeProduction(const shape_t& shape) {
  int64_t res = 1;
  for (auto i : shape) res *= i;
  return res;
}

void RunModel(std::string model_dir) {
   // 1. Create CxxConfig
   CxxConfig config;
   config.set_model_dir(model_dir);
   config.set_valid_places({
     Place{TARGET(kX86), PRECISION(kFloat)},
     Place{TARGET(kHost), PRECISION(kFloat)}
   });
  // 2. Create PaddlePredictor by CxxConfig
  std::shared_ptr<PaddlePredictor> predictor =
      CreatePaddlePredictor<CxxConfig>(config);

  // 3. Prepare input data
  std::unique_ptr<Tensor> input_tensor(std::move(predictor->GetInput(0)));
  input_tensor->Resize({1, 3, 224, 224});
  auto* data = input_tensor->mutable_data<float>();
  for (int i = 0; i < ShapeProduction(input_tensor->shape()); ++i) {
    data[i] = 1;
  }

  // 4. Run predictor
  predictor->Run();

  // 5. Get output
  std::unique_ptr<const Tensor> output_tensor(
      std::move(predictor->GetOutput(0)));
  std::cout << "Output shape " << output_tensor->shape()[1] << std::endl;
  for (int i = 0; i < ShapeProduction(output_tensor->shape()); i += 100) {
    std::cout << "Output[" << i << "]: " << output_tensor->data<float>()[i]
              << std::endl;
  }
}

int main(int argc, char** argv) {
  if (argc < 2) {
    std::cerr << "[ERROR] usage: ./" << argv[0] << " naive_buffer_model_dir\n";
    exit(1);
  }
  std::string model_dir = argv[1];
  RunModel(model_dir);
  return 0;
}

```

## 二、Windows环境

### 环境准备

#### 编译环境需求

- Windows 10 专业版
  - 目前Windows暂不支持GPU编译
- *Python 版本 2.7/3.5 (64 bit)*
- *pip 或 pip3 版本 9.0.1+ (64 bit)*
- *Visual Studio 2015 Update3*

#### 安装步骤

1. cmake 需要3.15版本, 可在官网[下载](https://cmake.org/download/)，并添加到环境变量中。

2. python 需要2.7 及以上版本, 可在官网[下载](https://www.python.org/download/releases/2.7/)。

3. git可以在官网[下载](https://gitforwindows.org/)，并添加到环境变量中

### 编译

1、 下载代码

```bash
git clone https://github.com/PaddlePaddle/Paddle-Lite.git
# 切换到release分支
git checkout release/v2.6.0
```
2、 源码编译(需要按照提示输入对应的参数)

```dos
cd Paddle-Lite
lite\tools\build_windows.bat 
```

### 编译结果说明

x86编译结果位于 `build.lite.x86/inference_lite_lib`
**具体内容**说明：

1、 `cxx`文件夹：包含c++的库文件与相应的头文件

- `include`  : 头文件
- `lib` : 库文件
  - 静态库文件：
    - `libpaddle_api_full_bundled.lib`  ：full_api 静态库
    - `libpaddle_api_light_bundled.lib` ：light_api 静态库

2、 `third_party` 文件夹：依赖的第三方预测库mklml

- mklml : Paddle-Lite预测库依赖的mklml数学库

3、 `demo/cxx`文件夹：x86预测库的C++ 示例demo

- `mobilenetv1_full` ：使用full_api 执行mobilenet_v1预测的C++ demo
- `mobilenetv1_light` ：使用light_api 执行mobilenet_v1预测的C++ demo

4、 `demo/python`文件夹：x86预测库的Python 示例demo

- `mobilenetv1_full_api.py` ：使用full_api 执行mobilenet_v1预测的Python demo
- `mobilenetv1_light_api.py` ：使用light_api 执行mobilenet_v1预测的Python demo

5、 `python`文件夹：包含python的库文件和对应的.whl包

- `install`文件夹：编译成功的.whl包位于`install/dist/*.whl`
- `lib`文件夹：.whl包依赖的库文件

### x86预测API使用示例

1、`mobilenetv1_full`目录结构

```bash
mobilenetv1_full/
|-- CMakeLists.txt
|-- build.sh
|-- build.bat
`-- mobilenet_full_api.cc
```

本demo使用cmake构建`CMakeLists.txt`为cmake脚本，`mobilenet_full_api.cc`是x86示例的源代码、`build.sh`为Linux x86编译的脚本，`build.bat`为windows x86编译脚本。

2、demo使用方法

``` bash
# 1、编译
cd mobilenetv1_full
build.bat
cd build 
```
编译结果为当前目录下的 `Release\mobilenet_full_api.exe `
``` dos
# 2、执行预测
Release\mobilenet_full_api.exe mobilenet_v1
```
下载并解压模型[`mobilenet_v1`](http://paddle-inference-dist.bj.bcebos.com/mobilenet_v1.tar.gz)到`build`目录，执行以上命令进行预测。
