# PaddleLite使用X86预测部署

Paddle-Lite 支持在Docker或Linux环境编译x86预测库。环境搭建参考[环境准备](../user_guides/source_compile)。

(注意：非docker Linux环境需要是Ubuntu16.04)

## 编译

1、 下载代码
```bash
git clone https://github.com/PaddlePaddle/Paddle-Lite.git
# 切换到release分支
git checkout release/v2.3
```

2、 源码编译

```bash
cd Paddle-Lite
./lite/tools/build.sh x86
```

## 编译结果说明

x86编译结果位于 `build.lite.x86/inference_lite_lib`
**具体内容**说明：

1、 `bin`文件夹：可执行工具文件 `test_model_bin`

2、 `cxx`文件夹：包含c++的库文件与相应的头文件

- `include`  : 头文件
- `lib` : 库文件
  - 打包的静态库文件：
    - `libpaddle_api_full_bundled.a`  ：包含 full_api 和 light_api 功能的静态库
    - `libpaddle_api_light_bundled.a` ：只包含 light_api 功能的静态库
  - 打包的动态态库文件：
    - `libpaddle_full_api_shared.so` ：包含 full_api 和 light_api 功能的动态库
    - `libpaddle_light_api_shared.so`：只包含 light_api 功能的动态库

3、 `third_party` 文件夹：第三方库文件

## x86预测API使用示例

1、我们提供Linux环境下x86 API运行mobilenet_v1的示例：[mobilenet_full_x86demo](https://paddlelite-data.bj.bcebos.com/x86/mobilenet_full_x86demo.zip)。下载解压后内容如下：

![](https://paddlelite-data.bj.bcebos.com/x86/x86-doc/demo.png)

`mobilenet_v1`为模型文件、`lib`和`include`分别是Paddle-Lite的预测库和头文件、`third_party`下是编译时依赖的第三方库`mklml`、`mobilenet_full_api.cc`是x86示例的源代码、`build.sh`为编译的脚本。

2、demo内容与使用方法

``` bash
# 1、编译
sh build.sh
```
编译结果为当前目录下的 `mobilenet_full_api `
``` bash
# 2、执行预测
mobilenet_full_api mobilenet_v1
```
`mobilenet_v1`为当前目录下的模型路径，`mobilenet_full_api`为第一步编译出的可执行文件。

3、示例源码`mobilenet_full_api.cc`

```c++
#include <iostream>
#include <vector>
#include "paddle_api.h"


using namespace paddle::lite_api;  // NOLINT

int64_t ShapeProduction(const shape_t& shape) {
  int64_t res = 1;
  for (auto i : shape) res *= i;
  return res;
}

void RunModel(std::string model_dir) {
   // 1. Create CxxConfig
   CxxConfig config;
   config.set_model_dir(model_dir);
   config.set_valid_places({
     Place{TARGET(kX86), PRECISION(kFloat)},
     Place{TARGET(kHost), PRECISION(kFloat)}
   });
  // 2. Create PaddlePredictor by CxxConfig
  std::shared_ptr<PaddlePredictor> predictor =
      CreatePaddlePredictor<CxxConfig>(config);

  // 3. Prepare input data
  std::unique_ptr<Tensor> input_tensor(std::move(predictor->GetInput(0)));
  input_tensor->Resize({1, 3, 224, 224});
  auto* data = input_tensor->mutable_data<float>();
  for (int i = 0; i < ShapeProduction(input_tensor->shape()); ++i) {
    data[i] = 1;
  }

  // 4. Run predictor
  predictor->Run();

  // 5. Get output
  std::unique_ptr<const Tensor> output_tensor(
      std::move(predictor->GetOutput(0)));
  std::cout << "Output shape " << output_tensor->shape()[1] << std::endl;
  for (int i = 0; i < ShapeProduction(output_tensor->shape()); i += 100) {
    std::cout << "Output[" << i << "]: " << output_tensor->data<float>()[i]
              << std::endl;
  }
}

int main(int argc, char** argv) {
  if (argc < 2) {
    std::cerr << "[ERROR] usage: ./" << argv[0] << " naive_buffer_model_dir\n";
    exit(1);
  }
  std::string model_dir = argv[1];
  RunModel(model_dir);
  return 0;
}

```
