# PaddleLite使用X86预测部署

Paddle-Lite 支持在Docker或Linux环境编译x86预测库。环境搭建参考[环境准备](../user_guides/source_compile)。

## 编译

1、 下载代码
```bash
# 下载Paddle-Lite源码
git clone https://github.com/PaddlePaddle/Paddle-Lite.git
# 切换到release分支
git checkout release/v2.6.0
```

2、 源码编译

```bash
cd Paddle-Lite
./lite/tools/build.sh x86

# 其他可选择编译选项
# --with_log=OFF 关闭LOG信息输出
```

## 编译结果说明

x86编译结果位于 `build.lite.x86/inference_lite_lib`
**具体内容**说明：

1、 `bin`文件夹：可执行工具文件 `test_model_bin`

2、 `cxx`文件夹：包含c++的库文件与相应的头文件

- `include`  : 头文件
- `lib` : 库文件
  - 静态库文件：
    - `libpaddle_api_full_bundled.a`  ：full_api 静态库
    - `libpaddle_api_light_bundled.a` ：light_api 静态库
  - 动态库文件：
    - `libpaddle_full_api_shared.so` ：full_api 动态库
    - `libpaddle_light_api_shared.so`：light_api 动态库

3、 `third_party` 文件夹：依赖的第三方预测库mklml

- mklml : Paddle-Lite预测库依赖的mklml数学库

4、 `demo/cxx`文件夹：x86预测库的C++ 示例demo

- `mobilenetv1_full` ：使用full_api 执行mobilenet_v1预测的C++ demo
- `mobilenetv1_light` ：使用light_api 执行mobilenet_v1预测的C++ demo




## x86预测API使用示例

1、`mobilenetv1_full`目录结构

```bash
mobilenetv1_full/
|-- CMakeLists.txt
|-- build.sh
`-- mobilenet_full_api.cc
```

本demo使用cmake构建`CMakeLists.txt`为cmake脚本，`mobilenet_full_api.cc`是x86示例的源代码、`build.sh`为编译的脚本。

2、demo使用方法

``` bash
# 1、编译
cd mobilenetv1_full
sh build.sh
```
编译结果为当前目录下的 `mobilenet_full_api `
``` bash
# 2、执行预测
./mobilenet_full_api ./mobilenet_v1
```
下载并解压模型[`mobilenet_v1`](http://paddle-inference-dist.bj.bcebos.com/mobilenet_v1.tar.gz)到当前目录，执行以上命令进行预测。

```bash
# 3、执行demo后输出结果如下，全一输入下mobilenet_v1的预测结果
Output shape 1000
Output[0]: 0.000191312
Output[100]: 0.000159713
Output[200]: 0.000264313
Output[300]: 0.000210793
Output[400]: 0.00103236
Output[500]: 0.000110071
Output[600]: 0.00482924
Output[700]: 0.00184533
Output[800]: 0.000202116
Output[900]: 0.000585591
```



3、示例源码`mobilenet_full_api.cc`

```c++
#include <iostream>
#include <vector>
#include "paddle_api.h"


using namespace paddle::lite_api;  // NOLINT

int64_t ShapeProduction(const shape_t& shape) {
  int64_t res = 1;
  for (auto i : shape) res *= i;
  return res;
}

void RunModel(std::string model_dir) {
   // 1. Create CxxConfig
   CxxConfig config;
   config.set_model_dir(model_dir);
   config.set_valid_places({
     Place{TARGET(kX86), PRECISION(kFloat)},
     Place{TARGET(kHost), PRECISION(kFloat)}
   });
  // 2. Create PaddlePredictor by CxxConfig
  std::shared_ptr<PaddlePredictor> predictor =
      CreatePaddlePredictor<CxxConfig>(config);

  // 3. Prepare input data
  std::unique_ptr<Tensor> input_tensor(std::move(predictor->GetInput(0)));
  input_tensor->Resize({1, 3, 224, 224});
  auto* data = input_tensor->mutable_data<float>();
  for (int i = 0; i < ShapeProduction(input_tensor->shape()); ++i) {
    data[i] = 1;
  }

  // 4. Run predictor
  predictor->Run();

  // 5. Get output
  std::unique_ptr<const Tensor> output_tensor(
      std::move(predictor->GetOutput(0)));
  std::cout << "Output shape " << output_tensor->shape()[1] << std::endl;
  for (int i = 0; i < ShapeProduction(output_tensor->shape()); i += 100) {
    std::cout << "Output[" << i << "]: " << output_tensor->data<float>()[i]
              << std::endl;
  }
}

int main(int argc, char** argv) {
  if (argc < 2) {
    std::cerr << "[ERROR] usage: ./" << argv[0] << " naive_buffer_model_dir\n";
    exit(1);
  }
  std::string model_dir = argv[1];
  RunModel(model_dir);
  return 0;
}

```
