/* Copyright (c) 2018 PaddlePaddle Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. */

#ifdef SOFTMAX_OP

#include "operators/kernel/softmax_kernel.h"
#include "operators/kernel/central-arm-func/softmax_arm_func.h"

namespace paddle_mobile {
namespace operators {

template <>
bool SoftmaxKernel<FPGA, float>::Init(SoftmaxParam<FPGA> *param) {
  auto input = const_cast<LoDTensor *>(param->InputX());
  auto dims = framework::vectorize(input->dims());
  half *input_ptr;
  auto out = param->Out();
  if (input->type() == typeid(float)) {
    out->Resize(framework::make_ddim(dims));
    out->mutable_data<float>(framework::make_ddim(dims));
  } else {
    input_ptr = input->data<half>();
  }

  auto float_input = new Tensor;

  PADDLE_MOBILE_ENFORCE(input->dims().size() == 4,
                        "Softmax should have 4-order input");

  auto channel = dims[3];
  if (channel == 1) {  // This input is generated by FC op, dims = [N C 1 1]
    PADDLE_MOBILE_ENFORCE(dims[2] == 1, "Softmax input must come from FC op");
    dims[3] = dims[1];
    dims[1] = 1;
  }
  input->Resize(framework::make_ddim(dims));
  float_input->Resize(framework::make_ddim(dims));

  if (channel != 2) {  // Use CPU
    out->Resize(framework::make_ddim(dims));
    out->mutable_data<float>(framework::make_ddim(dims));
    float_input->init(typeid(float));
    float_input->mutable_data<float>(framework::make_ddim(dims));
    //  fpga::format_fp32_ofm(float_input);
    // fpga::format_fp32_ofm(out);

    fpga::BypassArgs args = {fpga::DATA_TYPE_FP16};
    args.input_layout_type = fpga::LAYOUT_HWC;
    args.output_layout_type = fpga::LAYOUT_CHW;
    args.input_data_type = fpga::DATA_TYPE_FP16;
    args.output_data_type = fpga::DATA_TYPE_FP32;
    args.image.address = input_ptr;
    args.image.height = (uint32_t)dims[1] * dims[0];
    args.image.width = (uint32_t)dims[2];
    args.image.channels = (uint32_t)dims[3];
    args.output.address = float_input->data<float>();
    args.output.scale_address = float_input->scale;
    param->SetFloatInput(float_input);
    param->SetFpgaArgs(args);
  } else {  // Use FPGA
    fpga::format_fp16_ofm(out);
    fpga::BypassArgs args = {fpga::DATA_TYPE_FP16};
    args.input_layout_type = fpga::LAYOUT_HWC;
    args.output_layout_type = fpga::LAYOUT_CHW;
    args.input_data_type = fpga::DATA_TYPE_FP16;
    args.output_data_type = fpga::DATA_TYPE_FP16;
    args.image.address = input_ptr;
    args.image.height = (uint32_t)input->dims()[1];
    args.image.width = (uint32_t)input->dims()[2];
    args.image.channels = (uint32_t)input->dims()[3];
    args.output.address = out->data<half>();
    args.output.scale_address = out->scale;
    args.output.activation.activation_type = fpga::SOFTMAX;
    param->SetFpgaArgs(args);
  }

  return true;
}

template <>
void SoftmaxKernel<FPGA, float>::Compute(const SoftmaxParam<FPGA> &param) {
  auto *in_x = (param.InputX());
  if (in_x->type() == typeid(half)) {
    fpga::PerformBypass(param.FpgaArgs());
    if (param.FpgaArgs().output.activation.activation_type != fpga::SOFTMAX) {
      Tensor *out = param.Out();
      Tensor *in_x2 = param.FloatInput();

      fpga::fpga_invalidate(in_x2->data<float>(),
                            in_x2->numel() * sizeof(float));
      math::SoftmaxFuntor<CPU, float>()(in_x2, out);
      fpga::fpga_flush(out->data<float>(), out->memory_size());
    }
  } else {
    if (param.FpgaArgs().output.activation.activation_type != fpga::SOFTMAX) {
      Tensor *out = param.Out();
      math::SoftmaxFuntor<CPU, float>()(in_x, out);
    }
  }
}

}  // namespace operators
}  // namespace paddle_mobile

#endif
