/*
 * Support functions for OMAP GPIO
 *
 * Copyright (C) 2003-2005 Nokia Corporation
 * Written by Juha Yrjölä <juha.yrjola@nokia.com>
 *
 * Copyright (C) 2009 Texas Instruments
 * Added OMAP4 support - Santosh Shilimkar <santosh.shilimkar@ti.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/syscore_ops.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/device.h>
#include <linux/pm_runtime.h>
#include <linux/pm.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/gpio.h>
#include <linux/bitops.h>
#include <linux/platform_data/gpio-omap.h>

#define OFF_MODE	1
#define OMAP4_GPIO_DEBOUNCINGTIME_MASK 0xFF

static LIST_HEAD(omap_gpio_list);

struct gpio_regs {
	u32 irqenable1;
	u32 irqenable2;
	u32 wake_en;
	u32 ctrl;
	u32 oe;
	u32 leveldetect0;
	u32 leveldetect1;
	u32 risingdetect;
	u32 fallingdetect;
	u32 dataout;
	u32 debounce;
	u32 debounce_en;
};

struct gpio_bank {
	struct list_head node;
	void __iomem *base;
	int irq;
	u32 non_wakeup_gpios;
	u32 enabled_non_wakeup_gpios;
	struct gpio_regs context;
	u32 saved_datain;
	u32 level_mask;
	u32 toggle_mask;
	raw_spinlock_t lock;
	raw_spinlock_t wa_lock;
	struct gpio_chip chip;
	struct clk *dbck;
	u32 mod_usage;
	u32 irq_usage;
	u32 dbck_enable_mask;
	bool dbck_enabled;
	bool is_mpuio;
	bool dbck_flag;
	bool loses_context;
	bool context_valid;
	int stride;
	u32 width;
	int context_loss_count;
	int power_mode;
	bool workaround_enabled;

	void (*set_dataout)(struct gpio_bank *bank, unsigned gpio, int enable);
	int (*get_context_loss_count)(struct device *dev);

	struct omap_gpio_reg_offs *regs;
};

#define GPIO_MOD_CTRL_BIT	BIT(0)

#define BANK_USED(bank) (bank->mod_usage || bank->irq_usage)
#define LINE_USED(line, offset) (line & (BIT(offset)))

static void omap_gpio_unmask_irq(struct irq_data *d);

static inline struct gpio_bank *omap_irq_data_get_bank(struct irq_data *d)
{
	struct gpio_chip *chip = irq_data_get_irq_chip_data(d);
	return gpiochip_get_data(chip);
}

static void omap_set_gpio_direction(struct gpio_bank *bank, int gpio,
				    int is_input)
{
	void __iomem *reg = bank->base;
	u32 l;

	reg += bank->regs->direction;
	l = readl_relaxed(reg);
	if (is_input)
		l |= BIT(gpio);
	else
		l &= ~(BIT(gpio));
	writel_relaxed(l, reg);
	bank->context.oe = l;
}


/* set data out value using dedicate set/clear register */
static void omap_set_gpio_dataout_reg(struct gpio_bank *bank, unsigned offset,
				      int enable)
{
	void __iomem *reg = bank->base;
	u32 l = BIT(offset);

	if (enable) {
		reg += bank->regs->set_dataout;
		bank->context.dataout |= l;
	} else {
		reg += bank->regs->clr_dataout;
		bank->context.dataout &= ~l;
	}

	writel_relaxed(l, reg);
}

/* set data out value using mask register */
static void omap_set_gpio_dataout_mask(struct gpio_bank *bank, unsigned offset,
				       int enable)
{
	void __iomem *reg = bank->base + bank->regs->dataout;
	u32 gpio_bit = BIT(offset);
	u32 l;

	l = readl_relaxed(reg);
	if (enable)
		l |= gpio_bit;
	else
		l &= ~gpio_bit;
	writel_relaxed(l, reg);
	bank->context.dataout = l;
}

static int omap_get_gpio_datain(struct gpio_bank *bank, int offset)
{
	void __iomem *reg = bank->base + bank->regs->datain;

	return (readl_relaxed(reg) & (BIT(offset))) != 0;
}

static int omap_get_gpio_dataout(struct gpio_bank *bank, int offset)
{
	void __iomem *reg = bank->base + bank->regs->dataout;

	return (readl_relaxed(reg) & (BIT(offset))) != 0;
}

static inline void omap_gpio_rmw(void __iomem *base, u32 reg, u32 mask, bool set)
{
	int l = readl_relaxed(base + reg);

	if (set)
		l |= mask;
	else
		l &= ~mask;

	writel_relaxed(l, base + reg);
}

static inline void omap_gpio_dbck_enable(struct gpio_bank *bank)
{
	if (bank->dbck_enable_mask && !bank->dbck_enabled) {
		clk_enable(bank->dbck);
		bank->dbck_enabled = true;

		writel_relaxed(bank->dbck_enable_mask,
			     bank->base + bank->regs->debounce_en);
	}
}

static inline void omap_gpio_dbck_disable(struct gpio_bank *bank)
{
	if (bank->dbck_enable_mask && bank->dbck_enabled) {
		/*
		 * Disable debounce before cutting it's clock. If debounce is
		 * enabled but the clock is not, GPIO module seems to be unable
		 * to detect events and generate interrupts at least on OMAP3.
		 */
		writel_relaxed(0, bank->base + bank->regs->debounce_en);

		clk_disable(bank->dbck);
		bank->dbck_enabled = false;
	}
}

/**
 * omap2_set_gpio_debounce - low level gpio debounce time
 * @bank: the gpio bank we're acting upon
 * @offset: the gpio number on this @bank
 * @debounce: debounce time to use
 *
 * OMAP's debounce time is in 31us steps
 *   <debounce time> = (GPIO_DEBOUNCINGTIME[7:0].DEBOUNCETIME + 1) x 31
 * so we need to convert and round up to the closest unit.
 *
 * Return: 0 on success, negative error otherwise.
 */
static int omap2_set_gpio_debounce(struct gpio_bank *bank, unsigned offset,
				   unsigned debounce)
{
	void __iomem		*reg;
	u32			val;
	u32			l;
	bool			enable = !!debounce;

	if (!bank->dbck_flag)
		return -ENOTSUPP;

	if (enable) {
		debounce = DIV_ROUND_UP(debounce, 31) - 1;
		if ((debounce & OMAP4_GPIO_DEBOUNCINGTIME_MASK) != debounce)
			return -EINVAL;
	}

	l = BIT(offset);

	clk_enable(bank->dbck);
	reg = bank->base + bank->regs->debounce;
	writel_relaxed(debounce, reg);

	reg = bank->base + bank->regs->debounce_en;
	val = readl_relaxed(reg);

	if (enable)
		val |= l;
	else
		val &= ~l;
	bank->dbck_enable_mask = val;

	writel_relaxed(val, reg);
	clk_disable(bank->dbck);
	/*
	 * Enable debounce clock per module.
	 * This call is mandatory because in omap_gpio_request() when
	 * *_runtime_get_sync() is called,  _gpio_dbck_enable() within
	 * runtime callbck fails to turn on dbck because dbck_enable_mask
	 * used within _gpio_dbck_enable() is still not initialized at
	 * that point. Therefore we have to enable dbck here.
	 */
	omap_gpio_dbck_enable(bank);
	if (bank->dbck_enable_mask) {
		bank->context.debounce = debounce;
		bank->context.debounce_en = val;
	}

	return 0;
}

/**
 * omap_clear_gpio_debounce - clear debounce settings for a gpio
 * @bank: the gpio bank we're acting upon
 * @offset: the gpio number on this @bank
 *
 * If a gpio is using debounce, then clear the debounce enable bit and if
 * this is the only gpio in this bank using debounce, then clear the debounce
 * time too. The debounce clock will also be disabled when calling this function
 * if this is the only gpio in the bank using debounce.
 */
static void omap_clear_gpio_debounce(struct gpio_bank *bank, unsigned offset)
{
	u32 gpio_bit = BIT(offset);

	if (!bank->dbck_flag)
		return;

	if (!(bank->dbck_enable_mask & gpio_bit))
		return;

	bank->dbck_enable_mask &= ~gpio_bit;
	bank->context.debounce_en &= ~gpio_bit;
        writel_relaxed(bank->context.debounce_en,
		     bank->base + bank->regs->debounce_en);

	if (!bank->dbck_enable_mask) {
		bank->context.debounce = 0;
		writel_relaxed(bank->context.debounce, bank->base +
			     bank->regs->debounce);
		clk_disable(bank->dbck);
		bank->dbck_enabled = false;
	}
}

static inline void omap_set_gpio_trigger(struct gpio_bank *bank, int gpio,
						unsigned trigger)
{
	void __iomem *base = bank->base;
	u32 gpio_bit = BIT(gpio);

	omap_gpio_rmw(base, bank->regs->leveldetect0, gpio_bit,
		      trigger & IRQ_TYPE_LEVEL_LOW);
	omap_gpio_rmw(base, bank->regs->leveldetect1, gpio_bit,
		      trigger & IRQ_TYPE_LEVEL_HIGH);
	omap_gpio_rmw(base, bank->regs->risingdetect, gpio_bit,
		      trigger & IRQ_TYPE_EDGE_RISING);
	omap_gpio_rmw(base, bank->regs->fallingdetect, gpio_bit,
		      trigger & IRQ_TYPE_EDGE_FALLING);

	bank->context.leveldetect0 =
			readl_relaxed(bank->base + bank->regs->leveldetect0);
	bank->context.leveldetect1 =
			readl_relaxed(bank->base + bank->regs->leveldetect1);
	bank->context.risingdetect =
			readl_relaxed(bank->base + bank->regs->risingdetect);
	bank->context.fallingdetect =
			readl_relaxed(bank->base + bank->regs->fallingdetect);

	if (likely(!(bank->non_wakeup_gpios & gpio_bit))) {
		omap_gpio_rmw(base, bank->regs->wkup_en, gpio_bit, trigger != 0);
		bank->context.wake_en =
			readl_relaxed(bank->base + bank->regs->wkup_en);
	}

	/* This part needs to be executed always for OMAP{34xx, 44xx} */
	if (!bank->regs->irqctrl) {
		/* On omap24xx proceed only when valid GPIO bit is set */
		if (bank->non_wakeup_gpios) {
			if (!(bank->non_wakeup_gpios & gpio_bit))
				goto exit;
		}

		/*
		 * Log the edge gpio and manually trigger the IRQ
		 * after resume if the input level changes
		 * to avoid irq lost during PER RET/OFF mode
		 * Applies for omap2 non-wakeup gpio and all omap3 gpios
		 */
		if (trigger & IRQ_TYPE_EDGE_BOTH)
			bank->enabled_non_wakeup_gpios |= gpio_bit;
		else
			bank->enabled_non_wakeup_gpios &= ~gpio_bit;
	}

exit:
	bank->level_mask =
		readl_relaxed(bank->base + bank->regs->leveldetect0) |
		readl_relaxed(bank->base + bank->regs->leveldetect1);
}

#ifdef CONFIG_ARCH_OMAP1
/*
 * This only applies to chips that can't do both rising and falling edge
 * detection at once.  For all other chips, this function is a noop.
 */
static void omap_toggle_gpio_edge_triggering(struct gpio_bank *bank, int gpio)
{
	void __iomem *reg = bank->base;
	u32 l = 0;

	if (!bank->regs->irqctrl)
		return;

	reg += bank->regs->irqctrl;

	l = readl_relaxed(reg);
	if ((l >> gpio) & 1)
		l &= ~(BIT(gpio));
	else
		l |= BIT(gpio);

	writel_relaxed(l, reg);
}
#else
static void omap_toggle_gpio_edge_triggering(struct gpio_bank *bank, int gpio) {}
#endif

static int omap_set_gpio_triggering(struct gpio_bank *bank, int gpio,
				    unsigned trigger)
{
	void __iomem *reg = bank->base;
	void __iomem *base = bank->base;
	u32 l = 0;

	if (bank->regs->leveldetect0 && bank->regs->wkup_en) {
		omap_set_gpio_trigger(bank, gpio, trigger);
	} else if (bank->regs->irqctrl) {
		reg += bank->regs->irqctrl;

		l = readl_relaxed(reg);
		if ((trigger & IRQ_TYPE_SENSE_MASK) == IRQ_TYPE_EDGE_BOTH)
			bank->toggle_mask |= BIT(gpio);
		if (trigger & IRQ_TYPE_EDGE_RISING)
			l |= BIT(gpio);
		else if (trigger & IRQ_TYPE_EDGE_FALLING)
			l &= ~(BIT(gpio));
		else
			return -EINVAL;

		writel_relaxed(l, reg);
	} else if (bank->regs->edgectrl1) {
		if (gpio & 0x08)
			reg += bank->regs->edgectrl2;
		else
			reg += bank->regs->edgectrl1;

		gpio &= 0x07;
		l = readl_relaxed(reg);
		l &= ~(3 << (gpio << 1));
		if (trigger & IRQ_TYPE_EDGE_RISING)
			l |= 2 << (gpio << 1);
		if (trigger & IRQ_TYPE_EDGE_FALLING)
			l |= BIT(gpio << 1);

		/* Enable wake-up during idle for dynamic tick */
		omap_gpio_rmw(base, bank->regs->wkup_en, BIT(gpio), trigger);
		bank->context.wake_en =
			readl_relaxed(bank->base + bank->regs->wkup_en);
		writel_relaxed(l, reg);
	}
	return 0;
}

static void omap_enable_gpio_module(struct gpio_bank *bank, unsigned offset)
{
	if (bank->regs->pinctrl) {
		void __iomem *reg = bank->base + bank->regs->pinctrl;

		/* Claim the pin for MPU */
		writel_relaxed(readl_relaxed(reg) | (BIT(offset)), reg);
	}

	if (bank->regs->ctrl && !BANK_USED(bank)) {
		void __iomem *reg = bank->base + bank->regs->ctrl;
		u32 ctrl;

		ctrl = readl_relaxed(reg);
		/* Module is enabled, clocks are not gated */
		ctrl &= ~GPIO_MOD_CTRL_BIT;
		writel_relaxed(ctrl, reg);
		bank->context.ctrl = ctrl;
	}
}

static void omap_disable_gpio_module(struct gpio_bank *bank, unsigned offset)
{
	void __iomem *base = bank->base;

	if (bank->regs->wkup_en &&
	    !LINE_USED(bank->mod_usage, offset) &&
	    !LINE_USED(bank->irq_usage, offset)) {
		/* Disable wake-up during idle for dynamic tick */
		omap_gpio_rmw(base, bank->regs->wkup_en, BIT(offset), 0);
		bank->context.wake_en =
			readl_relaxed(bank->base + bank->regs->wkup_en);
	}

	if (bank->regs->ctrl && !BANK_USED(bank)) {
		void __iomem *reg = bank->base + bank->regs->ctrl;
		u32 ctrl;

		ctrl = readl_relaxed(reg);
		/* Module is disabled, clocks are gated */
		ctrl |= GPIO_MOD_CTRL_BIT;
		writel_relaxed(ctrl, reg);
		bank->context.ctrl = ctrl;
	}
}

static int omap_gpio_is_input(struct gpio_bank *bank, unsigned offset)
{
	void __iomem *reg = bank->base + bank->regs->direction;

	return readl_relaxed(reg) & BIT(offset);
}

static void omap_gpio_init_irq(struct gpio_bank *bank, unsigned offset)
{
	if (!LINE_USED(bank->mod_usage, offset)) {
		omap_enable_gpio_module(bank, offset);
		omap_set_gpio_direction(bank, offset, 1);
	}
	bank->irq_usage |= BIT(offset);
}

static int omap_gpio_irq_type(struct irq_data *d, unsigned type)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);
	int retval;
	unsigned long flags;
	unsigned offset = d->hwirq;

	if (type & ~IRQ_TYPE_SENSE_MASK)
		return -EINVAL;

	if (!bank->regs->leveldetect0 &&
		(type & (IRQ_TYPE_LEVEL_LOW|IRQ_TYPE_LEVEL_HIGH)))
		return -EINVAL;

	raw_spin_lock_irqsave(&bank->lock, flags);
	retval = omap_set_gpio_triggering(bank, offset, type);
	if (retval) {
		raw_spin_unlock_irqrestore(&bank->lock, flags);
		goto error;
	}
	omap_gpio_init_irq(bank, offset);
	if (!omap_gpio_is_input(bank, offset)) {
		raw_spin_unlock_irqrestore(&bank->lock, flags);
		retval = -EINVAL;
		goto error;
	}
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	if (type & (IRQ_TYPE_LEVEL_LOW | IRQ_TYPE_LEVEL_HIGH))
		irq_set_handler_locked(d, handle_level_irq);
	else if (type & (IRQ_TYPE_EDGE_FALLING | IRQ_TYPE_EDGE_RISING))
		irq_set_handler_locked(d, handle_edge_irq);

	return 0;

error:
	return retval;
}

static void omap_clear_gpio_irqbank(struct gpio_bank *bank, int gpio_mask)
{
	void __iomem *reg = bank->base;

	reg += bank->regs->irqstatus;
	writel_relaxed(gpio_mask, reg);

	/* Workaround for clearing DSP GPIO interrupts to allow retention */
	if (bank->regs->irqstatus2) {
		reg = bank->base + bank->regs->irqstatus2;
		writel_relaxed(gpio_mask, reg);
	}

	/* Flush posted write for the irq status to avoid spurious interrupts */
	readl_relaxed(reg);
}

static inline void omap_clear_gpio_irqstatus(struct gpio_bank *bank,
					     unsigned offset)
{
	omap_clear_gpio_irqbank(bank, BIT(offset));
}

static u32 omap_get_gpio_irqbank_mask(struct gpio_bank *bank)
{
	void __iomem *reg = bank->base;
	u32 l;
	u32 mask = (BIT(bank->width)) - 1;

	reg += bank->regs->irqenable;
	l = readl_relaxed(reg);
	if (bank->regs->irqenable_inv)
		l = ~l;
	l &= mask;
	return l;
}

static void omap_enable_gpio_irqbank(struct gpio_bank *bank, int gpio_mask)
{
	void __iomem *reg = bank->base;
	u32 l;

	if (bank->regs->set_irqenable) {
		reg += bank->regs->set_irqenable;
		l = gpio_mask;
		bank->context.irqenable1 |= gpio_mask;
	} else {
		reg += bank->regs->irqenable;
		l = readl_relaxed(reg);
		if (bank->regs->irqenable_inv)
			l &= ~gpio_mask;
		else
			l |= gpio_mask;
		bank->context.irqenable1 = l;
	}

	writel_relaxed(l, reg);
}

static void omap_disable_gpio_irqbank(struct gpio_bank *bank, int gpio_mask)
{
	void __iomem *reg = bank->base;
	u32 l;

	if (bank->regs->clr_irqenable) {
		reg += bank->regs->clr_irqenable;
		l = gpio_mask;
		bank->context.irqenable1 &= ~gpio_mask;
	} else {
		reg += bank->regs->irqenable;
		l = readl_relaxed(reg);
		if (bank->regs->irqenable_inv)
			l |= gpio_mask;
		else
			l &= ~gpio_mask;
		bank->context.irqenable1 = l;
	}

	writel_relaxed(l, reg);
}

static inline void omap_set_gpio_irqenable(struct gpio_bank *bank,
					   unsigned offset, int enable)
{
	if (enable)
		omap_enable_gpio_irqbank(bank, BIT(offset));
	else
		omap_disable_gpio_irqbank(bank, BIT(offset));
}

/* Use disable_irq_wake() and enable_irq_wake() functions from drivers */
static int omap_gpio_wake_enable(struct irq_data *d, unsigned int enable)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);

	return irq_set_irq_wake(bank->irq, enable);
}

static int omap_gpio_request(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_bank *bank = gpiochip_get_data(chip);
	unsigned long flags;

	/*
	 * If this is the first gpio_request for the bank,
	 * enable the bank module.
	 */
	if (!BANK_USED(bank))
		pm_runtime_get_sync(chip->parent);

	raw_spin_lock_irqsave(&bank->lock, flags);
	omap_enable_gpio_module(bank, offset);
	bank->mod_usage |= BIT(offset);
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static void omap_gpio_free(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_bank *bank = gpiochip_get_data(chip);
	unsigned long flags;

	raw_spin_lock_irqsave(&bank->lock, flags);
	bank->mod_usage &= ~(BIT(offset));
	if (!LINE_USED(bank->irq_usage, offset)) {
		omap_set_gpio_direction(bank, offset, 1);
		omap_clear_gpio_debounce(bank, offset);
	}
	omap_disable_gpio_module(bank, offset);
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	/*
	 * If this is the last gpio to be freed in the bank,
	 * disable the bank module.
	 */
	if (!BANK_USED(bank))
		pm_runtime_put(chip->parent);
}

/*
 * We need to unmask the GPIO bank interrupt as soon as possible to
 * avoid missing GPIO interrupts for other lines in the bank.
 * Then we need to mask-read-clear-unmask the triggered GPIO lines
 * in the bank to avoid missing nested interrupts for a GPIO line.
 * If we wait to unmask individual GPIO lines in the bank after the
 * line's interrupt handler has been run, we may miss some nested
 * interrupts.
 */
static irqreturn_t omap_gpio_irq_handler(int irq, void *gpiobank)
{
	void __iomem *isr_reg = NULL;
	u32 isr;
	unsigned int bit;
	struct gpio_bank *bank = gpiobank;
	unsigned long wa_lock_flags;
	unsigned long lock_flags;

	isr_reg = bank->base + bank->regs->irqstatus;
	if (WARN_ON(!isr_reg))
		goto exit;

	pm_runtime_get_sync(bank->chip.parent);

	while (1) {
		u32 isr_saved, level_mask = 0;
		u32 enabled;

		raw_spin_lock_irqsave(&bank->lock, lock_flags);

		enabled = omap_get_gpio_irqbank_mask(bank);
		isr_saved = isr = readl_relaxed(isr_reg) & enabled;

		if (bank->level_mask)
			level_mask = bank->level_mask & enabled;

		/* clear edge sensitive interrupts before handler(s) are
		called so that we don't miss any interrupt occurred while
		executing them */
		omap_disable_gpio_irqbank(bank, isr_saved & ~level_mask);
		omap_clear_gpio_irqbank(bank, isr_saved & ~level_mask);
		omap_enable_gpio_irqbank(bank, isr_saved & ~level_mask);

		raw_spin_unlock_irqrestore(&bank->lock, lock_flags);

		if (!isr)
			break;

		while (isr) {
			bit = __ffs(isr);
			isr &= ~(BIT(bit));

			raw_spin_lock_irqsave(&bank->lock, lock_flags);
			/*
			 * Some chips can't respond to both rising and falling
			 * at the same time.  If this irq was requested with
			 * both flags, we need to flip the ICR data for the IRQ
			 * to respond to the IRQ for the opposite direction.
			 * This will be indicated in the bank toggle_mask.
			 */
			if (bank->toggle_mask & (BIT(bit)))
				omap_toggle_gpio_edge_triggering(bank, bit);

			raw_spin_unlock_irqrestore(&bank->lock, lock_flags);

			raw_spin_lock_irqsave(&bank->wa_lock, wa_lock_flags);

			generic_handle_irq(irq_find_mapping(bank->chip.irqdomain,
							    bit));

			raw_spin_unlock_irqrestore(&bank->wa_lock,
						   wa_lock_flags);
		}
	}
exit:
	pm_runtime_put(bank->chip.parent);
	return IRQ_HANDLED;
}

static unsigned int omap_gpio_irq_startup(struct irq_data *d)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);
	unsigned long flags;
	unsigned offset = d->hwirq;

	raw_spin_lock_irqsave(&bank->lock, flags);

	if (!LINE_USED(bank->mod_usage, offset))
		omap_set_gpio_direction(bank, offset, 1);
	else if (!omap_gpio_is_input(bank, offset))
		goto err;
	omap_enable_gpio_module(bank, offset);
	bank->irq_usage |= BIT(offset);

	raw_spin_unlock_irqrestore(&bank->lock, flags);
	omap_gpio_unmask_irq(d);

	return 0;
err:
	raw_spin_unlock_irqrestore(&bank->lock, flags);
	return -EINVAL;
}

static void omap_gpio_irq_shutdown(struct irq_data *d)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);
	unsigned long flags;
	unsigned offset = d->hwirq;

	raw_spin_lock_irqsave(&bank->lock, flags);
	bank->irq_usage &= ~(BIT(offset));
	omap_set_gpio_irqenable(bank, offset, 0);
	omap_clear_gpio_irqstatus(bank, offset);
	omap_set_gpio_triggering(bank, offset, IRQ_TYPE_NONE);
	if (!LINE_USED(bank->mod_usage, offset))
		omap_clear_gpio_debounce(bank, offset);
	omap_disable_gpio_module(bank, offset);
	raw_spin_unlock_irqrestore(&bank->lock, flags);
}

static void omap_gpio_irq_bus_lock(struct irq_data *data)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(data);

	if (!BANK_USED(bank))
		pm_runtime_get_sync(bank->chip.parent);
}

static void gpio_irq_bus_sync_unlock(struct irq_data *data)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(data);

	/*
	 * If this is the last IRQ to be freed in the bank,
	 * disable the bank module.
	 */
	if (!BANK_USED(bank))
		pm_runtime_put(bank->chip.parent);
}

static void omap_gpio_ack_irq(struct irq_data *d)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);
	unsigned offset = d->hwirq;

	omap_clear_gpio_irqstatus(bank, offset);
}

static void omap_gpio_mask_irq(struct irq_data *d)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);
	unsigned offset = d->hwirq;
	unsigned long flags;

	raw_spin_lock_irqsave(&bank->lock, flags);
	omap_set_gpio_irqenable(bank, offset, 0);
	omap_set_gpio_triggering(bank, offset, IRQ_TYPE_NONE);
	raw_spin_unlock_irqrestore(&bank->lock, flags);
}

static void omap_gpio_unmask_irq(struct irq_data *d)
{
	struct gpio_bank *bank = omap_irq_data_get_bank(d);
	unsigned offset = d->hwirq;
	u32 trigger = irqd_get_trigger_type(d);
	unsigned long flags;

	raw_spin_lock_irqsave(&bank->lock, flags);
	if (trigger)
		omap_set_gpio_triggering(bank, offset, trigger);

	/* For level-triggered GPIOs, the clearing must be done after
	 * the HW source is cleared, thus after the handler has run */
	if (bank->level_mask & BIT(offset)) {
		omap_set_gpio_irqenable(bank, offset, 0);
		omap_clear_gpio_irqstatus(bank, offset);
	}

	omap_set_gpio_irqenable(bank, offset, 1);
	raw_spin_unlock_irqrestore(&bank->lock, flags);
}

/*---------------------------------------------------------------------*/

static int omap_mpuio_suspend_noirq(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct gpio_bank	*bank = platform_get_drvdata(pdev);
	void __iomem		*mask_reg = bank->base +
					OMAP_MPUIO_GPIO_MASKIT / bank->stride;
	unsigned long		flags;

	raw_spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(0xffff & ~bank->context.wake_en, mask_reg);
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static int omap_mpuio_resume_noirq(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct gpio_bank	*bank = platform_get_drvdata(pdev);
	void __iomem		*mask_reg = bank->base +
					OMAP_MPUIO_GPIO_MASKIT / bank->stride;
	unsigned long		flags;

	raw_spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(bank->context.wake_en, mask_reg);
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static const struct dev_pm_ops omap_mpuio_dev_pm_ops = {
	.suspend_noirq = omap_mpuio_suspend_noirq,
	.resume_noirq = omap_mpuio_resume_noirq,
};

/* use platform_driver for this. */
static struct platform_driver omap_mpuio_driver = {
	.driver		= {
		.name	= "mpuio",
		.pm	= &omap_mpuio_dev_pm_ops,
	},
};

static struct platform_device omap_mpuio_device = {
	.name		= "mpuio",
	.id		= -1,
	.dev = {
		.driver = &omap_mpuio_driver.driver,
	}
	/* could list the /proc/iomem resources */
};

static inline void omap_mpuio_init(struct gpio_bank *bank)
{
	platform_set_drvdata(&omap_mpuio_device, bank);

	if (platform_driver_register(&omap_mpuio_driver) == 0)
		(void) platform_device_register(&omap_mpuio_device);
}

/*---------------------------------------------------------------------*/

static int omap_gpio_get_direction(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_bank *bank;
	unsigned long flags;
	void __iomem *reg;
	int dir;

	bank = gpiochip_get_data(chip);
	reg = bank->base + bank->regs->direction;
	raw_spin_lock_irqsave(&bank->lock, flags);
	dir = !!(readl_relaxed(reg) & BIT(offset));
	raw_spin_unlock_irqrestore(&bank->lock, flags);
	return dir;
}

static int omap_gpio_input(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_bank *bank;
	unsigned long flags;

	bank = gpiochip_get_data(chip);
	raw_spin_lock_irqsave(&bank->lock, flags);
	omap_set_gpio_direction(bank, offset, 1);
	raw_spin_unlock_irqrestore(&bank->lock, flags);
	return 0;
}

static int omap_gpio_get(struct gpio_chip *chip, unsigned offset)
{
	struct gpio_bank *bank;

	bank = gpiochip_get_data(chip);

	if (omap_gpio_is_input(bank, offset))
		return omap_get_gpio_datain(bank, offset);
	else
		return omap_get_gpio_dataout(bank, offset);
}

static int omap_gpio_output(struct gpio_chip *chip, unsigned offset, int value)
{
	struct gpio_bank *bank;
	unsigned long flags;

	bank = gpiochip_get_data(chip);
	raw_spin_lock_irqsave(&bank->lock, flags);
	bank->set_dataout(bank, offset, value);
	omap_set_gpio_direction(bank, offset, 0);
	raw_spin_unlock_irqrestore(&bank->lock, flags);
	return 0;
}

static int omap_gpio_debounce(struct gpio_chip *chip, unsigned offset,
			      unsigned debounce)
{
	struct gpio_bank *bank;
	unsigned long flags;
	int ret;

	bank = gpiochip_get_data(chip);

	raw_spin_lock_irqsave(&bank->lock, flags);
	ret = omap2_set_gpio_debounce(bank, offset, debounce);
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	if (ret)
		dev_info(chip->parent,
			 "Could not set line %u debounce to %u microseconds (%d)",
			 offset, debounce, ret);

	return ret;
}

static int omap_gpio_set_config(struct gpio_chip *chip, unsigned offset,
				unsigned long config)
{
	u32 debounce;

	if (pinconf_to_config_param(config) != PIN_CONFIG_INPUT_DEBOUNCE)
		return -ENOTSUPP;

	debounce = pinconf_to_config_argument(config);
	return omap_gpio_debounce(chip, offset, debounce);
}

static void omap_gpio_set(struct gpio_chip *chip, unsigned offset, int value)
{
	struct gpio_bank *bank;
	unsigned long flags;

	bank = gpiochip_get_data(chip);
	raw_spin_lock_irqsave(&bank->lock, flags);
	bank->set_dataout(bank, offset, value);
	raw_spin_unlock_irqrestore(&bank->lock, flags);
}

/*---------------------------------------------------------------------*/

static void __init omap_gpio_show_rev(struct gpio_bank *bank)
{
	static bool called;
	u32 rev;

	if (called || bank->regs->revision == USHRT_MAX)
		return;

	rev = readw_relaxed(bank->base + bank->regs->revision);
	pr_info("OMAP GPIO hardware version %d.%d\n",
		(rev >> 4) & 0x0f, rev & 0x0f);

	called = true;
}

static void omap_gpio_mod_init(struct gpio_bank *bank)
{
	void __iomem *base = bank->base;
	u32 l = 0xffffffff;

	if (bank->width == 16)
		l = 0xffff;

	if (bank->is_mpuio) {
		writel_relaxed(l, bank->base + bank->regs->irqenable);
		return;
	}

	omap_gpio_rmw(base, bank->regs->irqenable, l,
		      bank->regs->irqenable_inv);
	omap_gpio_rmw(base, bank->regs->irqstatus, l,
		      !bank->regs->irqenable_inv);
	if (bank->regs->debounce_en)
		writel_relaxed(0, base + bank->regs->debounce_en);

	/* Save OE default value (0xffffffff) in the context */
	bank->context.oe = readl_relaxed(bank->base + bank->regs->direction);
	 /* Initialize interface clk ungated, module enabled */
	if (bank->regs->ctrl)
		writel_relaxed(0, base + bank->regs->ctrl);
}

static int omap_gpio_chip_init(struct gpio_bank *bank, struct irq_chip *irqc)
{
	static int gpio;
	int irq_base = 0;
	int ret;

	/*
	 * REVISIT eventually switch from OMAP-specific gpio structs
	 * over to the generic ones
	 */
	bank->chip.request = omap_gpio_request;
	bank->chip.free = omap_gpio_free;
	bank->chip.get_direction = omap_gpio_get_direction;
	bank->chip.direction_input = omap_gpio_input;
	bank->chip.get = omap_gpio_get;
	bank->chip.direction_output = omap_gpio_output;
	bank->chip.set_config = omap_gpio_set_config;
	bank->chip.set = omap_gpio_set;
	if (bank->is_mpuio) {
		bank->chip.label = "mpuio";
		if (bank->regs->wkup_en)
			bank->chip.parent = &omap_mpuio_device.dev;
		bank->chip.base = OMAP_MPUIO(0);
	} else {
		bank->chip.label = "gpio";
		bank->chip.base = gpio;
	}
	bank->chip.ngpio = bank->width;

	ret = gpiochip_add_data(&bank->chip, bank);
	if (ret) {
		dev_err(bank->chip.parent,
			"Could not register gpio chip %d\n", ret);
		return ret;
	}

	if (!bank->is_mpuio)
		gpio += bank->width;

#ifdef CONFIG_ARCH_OMAP1
	/*
	 * REVISIT: Once we have OMAP1 supporting SPARSE_IRQ, we can drop
	 * irq_alloc_descs() since a base IRQ offset will no longer be needed.
	 */
	irq_base = devm_irq_alloc_descs(bank->chip.parent,
					-1, 0, bank->width, 0);
	if (irq_base < 0) {
		dev_err(bank->chip.parent, "Couldn't allocate IRQ numbers\n");
		return -ENODEV;
	}
#endif

	/* MPUIO is a bit different, reading IRQ status clears it */
	if (bank->is_mpuio) {
		irqc->irq_ack = dummy_irq_chip.irq_ack;
		if (!bank->regs->wkup_en)
			irqc->irq_set_wake = NULL;
	}

	ret = gpiochip_irqchip_add(&bank->chip, irqc,
				   irq_base, handle_bad_irq,
				   IRQ_TYPE_NONE);

	if (ret) {
		dev_err(bank->chip.parent,
			"Couldn't add irqchip to gpiochip %d\n", ret);
		gpiochip_remove(&bank->chip);
		return -ENODEV;
	}

	gpiochip_set_chained_irqchip(&bank->chip, irqc, bank->irq, NULL);

	ret = devm_request_irq(bank->chip.parent, bank->irq,
			       omap_gpio_irq_handler,
			       0, dev_name(bank->chip.parent), bank);
	if (ret)
		gpiochip_remove(&bank->chip);

	return ret;
}

static const struct of_device_id omap_gpio_match[];

static int omap_gpio_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *node = dev->of_node;
	const struct of_device_id *match;
	const struct omap_gpio_platform_data *pdata;
	struct resource *res;
	struct gpio_bank *bank;
	struct irq_chip *irqc;
	int ret;

	match = of_match_device(of_match_ptr(omap_gpio_match), dev);

	pdata = match ? match->data : dev_get_platdata(dev);
	if (!pdata)
		return -EINVAL;

	bank = devm_kzalloc(dev, sizeof(struct gpio_bank), GFP_KERNEL);
	if (!bank) {
		dev_err(dev, "Memory alloc failed\n");
		return -ENOMEM;
	}

	irqc = devm_kzalloc(dev, sizeof(*irqc), GFP_KERNEL);
	if (!irqc)
		return -ENOMEM;

	irqc->irq_startup = omap_gpio_irq_startup,
	irqc->irq_shutdown = omap_gpio_irq_shutdown,
	irqc->irq_ack = omap_gpio_ack_irq,
	irqc->irq_mask = omap_gpio_mask_irq,
	irqc->irq_unmask = omap_gpio_unmask_irq,
	irqc->irq_set_type = omap_gpio_irq_type,
	irqc->irq_set_wake = omap_gpio_wake_enable,
	irqc->irq_bus_lock = omap_gpio_irq_bus_lock,
	irqc->irq_bus_sync_unlock = gpio_irq_bus_sync_unlock,
	irqc->name = dev_name(&pdev->dev);
	irqc->flags = IRQCHIP_MASK_ON_SUSPEND;

	bank->irq = platform_get_irq(pdev, 0);
	if (bank->irq <= 0) {
		if (!bank->irq)
			bank->irq = -ENXIO;
		if (bank->irq != -EPROBE_DEFER)
			dev_err(dev,
				"can't get irq resource ret=%d\n", bank->irq);
		return bank->irq;
	}

	bank->chip.parent = dev;
	bank->chip.owner = THIS_MODULE;
	bank->dbck_flag = pdata->dbck_flag;
	bank->stride = pdata->bank_stride;
	bank->width = pdata->bank_width;
	bank->is_mpuio = pdata->is_mpuio;
	bank->non_wakeup_gpios = pdata->non_wakeup_gpios;
	bank->regs = pdata->regs;
#ifdef CONFIG_OF_GPIO
	bank->chip.of_node = of_node_get(node);
#endif
	if (node) {
		if (!of_property_read_bool(node, "ti,gpio-always-on"))
			bank->loses_context = true;
	} else {
		bank->loses_context = pdata->loses_context;

		if (bank->loses_context)
			bank->get_context_loss_count =
				pdata->get_context_loss_count;
	}

	if (bank->regs->set_dataout && bank->regs->clr_dataout)
		bank->set_dataout = omap_set_gpio_dataout_reg;
	else
		bank->set_dataout = omap_set_gpio_dataout_mask;

	raw_spin_lock_init(&bank->lock);
	raw_spin_lock_init(&bank->wa_lock);

	/* Static mapping, never released */
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	bank->base = devm_ioremap_resource(dev, res);
	if (IS_ERR(bank->base)) {
		return PTR_ERR(bank->base);
	}

	if (bank->dbck_flag) {
		bank->dbck = devm_clk_get(dev, "dbclk");
		if (IS_ERR(bank->dbck)) {
			dev_err(dev,
				"Could not get gpio dbck. Disable debounce\n");
			bank->dbck_flag = false;
		} else {
			clk_prepare(bank->dbck);
		}
	}

	platform_set_drvdata(pdev, bank);

	pm_runtime_enable(dev);
	pm_runtime_irq_safe(dev);
	pm_runtime_get_sync(dev);

	if (bank->is_mpuio)
		omap_mpuio_init(bank);

	omap_gpio_mod_init(bank);

	ret = omap_gpio_chip_init(bank, irqc);
	if (ret) {
		pm_runtime_put_sync(dev);
		pm_runtime_disable(dev);
		if (bank->dbck_flag)
			clk_unprepare(bank->dbck);
		return ret;
	}

	omap_gpio_show_rev(bank);

	pm_runtime_put(dev);

	list_add_tail(&bank->node, &omap_gpio_list);

	return 0;
}

static int omap_gpio_remove(struct platform_device *pdev)
{
	struct gpio_bank *bank = platform_get_drvdata(pdev);

	list_del(&bank->node);
	gpiochip_remove(&bank->chip);
	pm_runtime_disable(&pdev->dev);
	if (bank->dbck_flag)
		clk_unprepare(bank->dbck);

	return 0;
}

#ifdef CONFIG_ARCH_OMAP2PLUS

#if defined(CONFIG_PM)
static void omap_gpio_restore_context(struct gpio_bank *bank);

static int omap_gpio_runtime_suspend(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct gpio_bank *bank = platform_get_drvdata(pdev);
	u32 l1 = 0, l2 = 0;
	unsigned long flags;
	u32 wake_low, wake_hi;

	raw_spin_lock_irqsave(&bank->lock, flags);

	/*
	 * Only edges can generate a wakeup event to the PRCM.
	 *
	 * Therefore, ensure any wake-up capable GPIOs have
	 * edge-detection enabled before going idle to ensure a wakeup
	 * to the PRCM is generated on a GPIO transition. (c.f. 34xx
	 * NDA TRM 25.5.3.1)
	 *
	 * The normal values will be restored upon ->runtime_resume()
	 * by writing back the values saved in bank->context.
	 */
	wake_low = bank->context.leveldetect0 & bank->context.wake_en;
	if (wake_low)
		writel_relaxed(wake_low | bank->context.fallingdetect,
			     bank->base + bank->regs->fallingdetect);
	wake_hi = bank->context.leveldetect1 & bank->context.wake_en;
	if (wake_hi)
		writel_relaxed(wake_hi | bank->context.risingdetect,
			     bank->base + bank->regs->risingdetect);

	if (!bank->enabled_non_wakeup_gpios)
		goto update_gpio_context_count;

	if (bank->power_mode != OFF_MODE) {
		bank->power_mode = 0;
		goto update_gpio_context_count;
	}
	/*
	 * If going to OFF, remove triggering for all
	 * non-wakeup GPIOs.  Otherwise spurious IRQs will be
	 * generated.  See OMAP2420 Errata item 1.101.
	 */
	bank->saved_datain = readl_relaxed(bank->base +
						bank->regs->datain);
	l1 = bank->context.fallingdetect;
	l2 = bank->context.risingdetect;

	l1 &= ~bank->enabled_non_wakeup_gpios;
	l2 &= ~bank->enabled_non_wakeup_gpios;

	writel_relaxed(l1, bank->base + bank->regs->fallingdetect);
	writel_relaxed(l2, bank->base + bank->regs->risingdetect);

	bank->workaround_enabled = true;

update_gpio_context_count:
	if (bank->get_context_loss_count)
		bank->context_loss_count =
				bank->get_context_loss_count(dev);

	omap_gpio_dbck_disable(bank);
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static void omap_gpio_init_context(struct gpio_bank *p);

static int omap_gpio_runtime_resume(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct gpio_bank *bank = platform_get_drvdata(pdev);
	u32 l = 0, gen, gen0, gen1;
	unsigned long flags;
	int c;

	raw_spin_lock_irqsave(&bank->lock, flags);

	/*
	 * On the first resume during the probe, the context has not
	 * been initialised and so initialise it now. Also initialise
	 * the context loss count.
	 */
	if (bank->loses_context && !bank->context_valid) {
		omap_gpio_init_context(bank);

		if (bank->get_context_loss_count)
			bank->context_loss_count =
				bank->get_context_loss_count(dev);
	}

	omap_gpio_dbck_enable(bank);

	/*
	 * In ->runtime_suspend(), level-triggered, wakeup-enabled
	 * GPIOs were set to edge trigger also in order to be able to
	 * generate a PRCM wakeup.  Here we restore the
	 * pre-runtime_suspend() values for edge triggering.
	 */
	writel_relaxed(bank->context.fallingdetect,
		     bank->base + bank->regs->fallingdetect);
	writel_relaxed(bank->context.risingdetect,
		     bank->base + bank->regs->risingdetect);

	if (bank->loses_context) {
		if (!bank->get_context_loss_count) {
			omap_gpio_restore_context(bank);
		} else {
			c = bank->get_context_loss_count(dev);
			if (c != bank->context_loss_count) {
				omap_gpio_restore_context(bank);
			} else {
				raw_spin_unlock_irqrestore(&bank->lock, flags);
				return 0;
			}
		}
	}

	if (!bank->workaround_enabled) {
		raw_spin_unlock_irqrestore(&bank->lock, flags);
		return 0;
	}

	l = readl_relaxed(bank->base + bank->regs->datain);

	/*
	 * Check if any of the non-wakeup interrupt GPIOs have changed
	 * state.  If so, generate an IRQ by software.  This is
	 * horribly racy, but it's the best we can do to work around
	 * this silicon bug.
	 */
	l ^= bank->saved_datain;
	l &= bank->enabled_non_wakeup_gpios;

	/*
	 * No need to generate IRQs for the rising edge for gpio IRQs
	 * configured with falling edge only; and vice versa.
	 */
	gen0 = l & bank->context.fallingdetect;
	gen0 &= bank->saved_datain;

	gen1 = l & bank->context.risingdetect;
	gen1 &= ~(bank->saved_datain);

	/* FIXME: Consider GPIO IRQs with level detections properly! */
	gen = l & (~(bank->context.fallingdetect) &
					 ~(bank->context.risingdetect));
	/* Consider all GPIO IRQs needed to be updated */
	gen |= gen0 | gen1;

	if (gen) {
		u32 old0, old1;

		old0 = readl_relaxed(bank->base + bank->regs->leveldetect0);
		old1 = readl_relaxed(bank->base + bank->regs->leveldetect1);

		if (!bank->regs->irqstatus_raw0) {
			writel_relaxed(old0 | gen, bank->base +
						bank->regs->leveldetect0);
			writel_relaxed(old1 | gen, bank->base +
						bank->regs->leveldetect1);
		}

		if (bank->regs->irqstatus_raw0) {
			writel_relaxed(old0 | l, bank->base +
						bank->regs->leveldetect0);
			writel_relaxed(old1 | l, bank->base +
						bank->regs->leveldetect1);
		}
		writel_relaxed(old0, bank->base + bank->regs->leveldetect0);
		writel_relaxed(old1, bank->base + bank->regs->leveldetect1);
	}

	bank->workaround_enabled = false;
	raw_spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}
#endif /* CONFIG_PM */

#if IS_BUILTIN(CONFIG_GPIO_OMAP)
void omap2_gpio_prepare_for_idle(int pwr_mode)
{
	struct gpio_bank *bank;

	list_for_each_entry(bank, &omap_gpio_list, node) {
		if (!BANK_USED(bank) || !bank->loses_context)
			continue;

		bank->power_mode = pwr_mode;

		pm_runtime_put_sync_suspend(bank->chip.parent);
	}
}

void omap2_gpio_resume_after_idle(void)
{
	struct gpio_bank *bank;

	list_for_each_entry(bank, &omap_gpio_list, node) {
		if (!BANK_USED(bank) || !bank->loses_context)
			continue;

		pm_runtime_get_sync(bank->chip.parent);
	}
}
#endif

#if defined(CONFIG_PM)
static void omap_gpio_init_context(struct gpio_bank *p)
{
	struct omap_gpio_reg_offs *regs = p->regs;
	void __iomem *base = p->base;

	p->context.ctrl		= readl_relaxed(base + regs->ctrl);
	p->context.oe		= readl_relaxed(base + regs->direction);
	p->context.wake_en	= readl_relaxed(base + regs->wkup_en);
	p->context.leveldetect0	= readl_relaxed(base + regs->leveldetect0);
	p->context.leveldetect1	= readl_relaxed(base + regs->leveldetect1);
	p->context.risingdetect	= readl_relaxed(base + regs->risingdetect);
	p->context.fallingdetect = readl_relaxed(base + regs->fallingdetect);
	p->context.irqenable1	= readl_relaxed(base + regs->irqenable);
	p->context.irqenable2	= readl_relaxed(base + regs->irqenable2);

	if (regs->set_dataout && p->regs->clr_dataout)
		p->context.dataout = readl_relaxed(base + regs->set_dataout);
	else
		p->context.dataout = readl_relaxed(base + regs->dataout);

	p->context_valid = true;
}

static void omap_gpio_restore_context(struct gpio_bank *bank)
{
	writel_relaxed(bank->context.wake_en,
				bank->base + bank->regs->wkup_en);
	writel_relaxed(bank->context.ctrl, bank->base + bank->regs->ctrl);
	writel_relaxed(bank->context.leveldetect0,
				bank->base + bank->regs->leveldetect0);
	writel_relaxed(bank->context.leveldetect1,
				bank->base + bank->regs->leveldetect1);
	writel_relaxed(bank->context.risingdetect,
				bank->base + bank->regs->risingdetect);
	writel_relaxed(bank->context.fallingdetect,
				bank->base + bank->regs->fallingdetect);
	if (bank->regs->set_dataout && bank->regs->clr_dataout)
		writel_relaxed(bank->context.dataout,
				bank->base + bank->regs->set_dataout);
	else
		writel_relaxed(bank->context.dataout,
				bank->base + bank->regs->dataout);
	writel_relaxed(bank->context.oe, bank->base + bank->regs->direction);

	if (bank->dbck_enable_mask) {
		writel_relaxed(bank->context.debounce, bank->base +
					bank->regs->debounce);
		writel_relaxed(bank->context.debounce_en,
					bank->base + bank->regs->debounce_en);
	}

	writel_relaxed(bank->context.irqenable1,
				bank->base + bank->regs->irqenable);
	writel_relaxed(bank->context.irqenable2,
				bank->base + bank->regs->irqenable2);
}
#endif /* CONFIG_PM */
#else
#define omap_gpio_runtime_suspend NULL
#define omap_gpio_runtime_resume NULL
static inline void omap_gpio_init_context(struct gpio_bank *p) {}
#endif

static const struct dev_pm_ops gpio_pm_ops = {
	SET_RUNTIME_PM_OPS(omap_gpio_runtime_suspend, omap_gpio_runtime_resume,
									NULL)
};

#if defined(CONFIG_OF)
static struct omap_gpio_reg_offs omap2_gpio_regs = {
	.revision =		OMAP24XX_GPIO_REVISION,
	.direction =		OMAP24XX_GPIO_OE,
	.datain =		OMAP24XX_GPIO_DATAIN,
	.dataout =		OMAP24XX_GPIO_DATAOUT,
	.set_dataout =		OMAP24XX_GPIO_SETDATAOUT,
	.clr_dataout =		OMAP24XX_GPIO_CLEARDATAOUT,
	.irqstatus =		OMAP24XX_GPIO_IRQSTATUS1,
	.irqstatus2 =		OMAP24XX_GPIO_IRQSTATUS2,
	.irqenable =		OMAP24XX_GPIO_IRQENABLE1,
	.irqenable2 =		OMAP24XX_GPIO_IRQENABLE2,
	.set_irqenable =	OMAP24XX_GPIO_SETIRQENABLE1,
	.clr_irqenable =	OMAP24XX_GPIO_CLEARIRQENABLE1,
	.debounce =		OMAP24XX_GPIO_DEBOUNCE_VAL,
	.debounce_en =		OMAP24XX_GPIO_DEBOUNCE_EN,
	.ctrl =			OMAP24XX_GPIO_CTRL,
	.wkup_en =		OMAP24XX_GPIO_WAKE_EN,
	.leveldetect0 =		OMAP24XX_GPIO_LEVELDETECT0,
	.leveldetect1 =		OMAP24XX_GPIO_LEVELDETECT1,
	.risingdetect =		OMAP24XX_GPIO_RISINGDETECT,
	.fallingdetect =	OMAP24XX_GPIO_FALLINGDETECT,
};

static struct omap_gpio_reg_offs omap4_gpio_regs = {
	.revision =		OMAP4_GPIO_REVISION,
	.direction =		OMAP4_GPIO_OE,
	.datain =		OMAP4_GPIO_DATAIN,
	.dataout =		OMAP4_GPIO_DATAOUT,
	.set_dataout =		OMAP4_GPIO_SETDATAOUT,
	.clr_dataout =		OMAP4_GPIO_CLEARDATAOUT,
	.irqstatus =		OMAP4_GPIO_IRQSTATUS0,
	.irqstatus2 =		OMAP4_GPIO_IRQSTATUS1,
	.irqenable =		OMAP4_GPIO_IRQSTATUSSET0,
	.irqenable2 =		OMAP4_GPIO_IRQSTATUSSET1,
	.set_irqenable =	OMAP4_GPIO_IRQSTATUSSET0,
	.clr_irqenable =	OMAP4_GPIO_IRQSTATUSCLR0,
	.debounce =		OMAP4_GPIO_DEBOUNCINGTIME,
	.debounce_en =		OMAP4_GPIO_DEBOUNCENABLE,
	.ctrl =			OMAP4_GPIO_CTRL,
	.wkup_en =		OMAP4_GPIO_IRQWAKEN0,
	.leveldetect0 =		OMAP4_GPIO_LEVELDETECT0,
	.leveldetect1 =		OMAP4_GPIO_LEVELDETECT1,
	.risingdetect =		OMAP4_GPIO_RISINGDETECT,
	.fallingdetect =	OMAP4_GPIO_FALLINGDETECT,
};

static const struct omap_gpio_platform_data omap2_pdata = {
	.regs = &omap2_gpio_regs,
	.bank_width = 32,
	.dbck_flag = false,
};

static const struct omap_gpio_platform_data omap3_pdata = {
	.regs = &omap2_gpio_regs,
	.bank_width = 32,
	.dbck_flag = true,
};

static const struct omap_gpio_platform_data omap4_pdata = {
	.regs = &omap4_gpio_regs,
	.bank_width = 32,
	.dbck_flag = true,
};

static const struct of_device_id omap_gpio_match[] = {
	{
		.compatible = "ti,omap4-gpio",
		.data = &omap4_pdata,
	},
	{
		.compatible = "ti,omap3-gpio",
		.data = &omap3_pdata,
	},
	{
		.compatible = "ti,omap2-gpio",
		.data = &omap2_pdata,
	},
	{ },
};
MODULE_DEVICE_TABLE(of, omap_gpio_match);
#endif

static struct platform_driver omap_gpio_driver = {
	.probe		= omap_gpio_probe,
	.remove		= omap_gpio_remove,
	.driver		= {
		.name	= "omap_gpio",
		.pm	= &gpio_pm_ops,
		.of_match_table = of_match_ptr(omap_gpio_match),
	},
};

/*
 * gpio driver register needs to be done before
 * machine_init functions access gpio APIs.
 * Hence omap_gpio_drv_reg() is a postcore_initcall.
 */
static int __init omap_gpio_drv_reg(void)
{
	return platform_driver_register(&omap_gpio_driver);
}
postcore_initcall(omap_gpio_drv_reg);

static void __exit omap_gpio_exit(void)
{
	platform_driver_unregister(&omap_gpio_driver);
}
module_exit(omap_gpio_exit);

MODULE_DESCRIPTION("omap gpio driver");
MODULE_ALIAS("platform:gpio-omap");
MODULE_LICENSE("GPL v2");
