# Time Management<a name="EN-US_TOPIC_0000001078876462"></a>



## Basic Concepts<a name="section36251149131120"></a>

Time management provides all time-related services for applications based on the system clock.

The system clock is generated by the interrupts triggered by the output pulse of a timer or counter. The system clock is generally defined as an integer or a long integer. The period of an output pulse is a "clock tick". The system clock is also called time scale or tick.

People use second or millisecond as the time unit, while the operating system uses tick. When operations such as suspending a task or delaying a task are performed, the time management module converts time between ticks and seconds or milliseconds.

The time management module of the OpenHarmony LiteOS-M kernel provides time conversion and statistics functions.

## Time Unit<a name="section97172532397"></a>

-   Cycle

    Cycle is the minimum time unit in the system. The cycle duration is determined by the system clock frequency, that is, the number of cycles per second.

-   Tick

    Tick is the basic time unit of the operating system and is determined by the number of ticks per second configured by the user.


## Available APIs<a name="section158501652121514"></a>

The following table describes APIs available for the OpenHarmony LiteOS-M time management. For more details about the APIs, see the API reference.

**Table  1**  APIs of the time management module

| Category| API| Description|
| -------- | -------- | -------- |
| Time conversion| LOS_MS2Tick | Converts milliseconds into ticks.|
|  | LOS_Tick2MS | Converts ticks into milliseconds.|
|  | OsCpuTick2MS | Converts cycles into milliseconds. Two UINT32 values indicate the high-order and low-order 32 bits of the result value, respectively.|
|  | OsCpuTick2US | Converts cycles into microseconds. Two UINT32 values indicate the high-order and low-order 32 bits of the result value, respectively.|
| Time statistics| LOS_SysClockGet | Obtains the system clock.|
|  | LOS_TickCountGet | Obtains the number of ticks since the system starts.|
|  | LOS_CyclePerTickGet | Obtains the number of cycles for each tick.|
|  | LOS_CurrNanosec |Obtains the number of nanoseconds since the system starts.|
| Delay management| LOS_UDelay |Performs busy waiting in μs, which can be preempted by a task with a higher priority.|
|  | LOS_MDelay |Performs busy waiting in ms, which can be preempted by a task with a higher priority.|

## How to Develop<a name="section783435801510"></a>

The typical development process of time management is as follows:

1.  Complete board configuration and adaptation as required, and configure the system clock frequency \(**OS\_SYS\_CLOCK**  in Hz and  **LOSCFG\_BASE\_CORE\_TICK\_PER\_SECOND**\). The default value of  **OS\_SYS\_CLOCK**  varies with the hardware platform.
2.  Call the clock conversion and statistics APIs.

>![](../public_sys-resources/icon-note.gif) **NOTE** 
>
>-   The time management module depends on  **OS\_SYS\_CLOCK**  and  **LOSCFG\_BASE\_CORE\_TICK\_PER\_SECOND**.
>-   The number of system ticks is not counted when the interrupt feature is disabled. Therefore, the number of ticks cannot be used as the accurate time.
>-   The configuration options are maintained in the  **target\_config.h**  file of the development board project.

## Development Example<a name="section460018317164"></a>

### Example Description<a name="section127752801718"></a>

The following example describes basic time management methods, including:

- Time conversion: convert milliseconds to ticks or convert ticks to milliseconds.
- Time statistics: obtain the number of cycles per tick, number of ticks since system startup, and number of delayed ticks.

### Sample Code<a name="section321653551711"></a>

Prerequisites

-   The default value of  **LOSCFG\_BASE\_CORE\_TICK\_PER\_SECOND**  is  **100**.
-   The system clock frequency  **OS\_SYS\_CLOCK**  is configured.

Time conversion:

```
VOID Example_TransformTime(VOID)
{
    UINT32 ms;
    UINT32 tick;

    tick = LOS_MS2Tick(10000);    // Convert 10000 ms into ticks.
    dprintf("tick = %d \n", tick);
    ms = LOS_Tick2MS(100);        // Convert 100 ticks into ms.
    dprintf("ms = %d \n", ms);
}
```

Time statistics and delay:

```
VOID Example_GetTime(VOID)
{
    UINT32 cyclePerTick;
    UINT64 tickCount;

    cyclePerTick  = LOS_CyclePerTickGet();
    if(0 != cyclePerTick) {
        dprintf("LOS_CyclePerTickGet = %d \n", cyclePerTick);
    }

    tickCount = LOS_TickCountGet();
    if(0 != tickCount) {
        dprintf("LOS_TickCountGet = %d \n", (UINT32)tickCount);
    }

    LOS_TaskDelay(200);
    tickCount = LOS_TickCountGet();
    if(0 != tickCount) {
        dprintf("LOS_TickCountGet after delay = %d \n", (UINT32)tickCount);
    }
}
```

### Verification<a name="section4366193318167"></a>

The development is successful if the return result is as follows:

Time conversion:

```
tick = 1000
ms = 1000
```

Time statistics and delay:

```
LOS_CyclePerTickGet = 495000 
LOS_TickCountGet = 1 
LOS_TickCountGet after delay = 201
```
