# Time Management


## Basic Concepts

Time management is performed based on the system clock. It provides time-related services for applications. The system clock is generated by the interrupts triggered by the output pulse of a timer or counter. The system clock is generally defined as an integer or a long integer. The period of an output pulse is a "clock tick".

The system clock is also called time scale or tick. The duration of a tick can be configured statically. People use second or millisecond as the time unit, while the operating system uses tick. When operations such as suspending a task or delaying a task are performed, the time management module converts time between ticks and seconds or milliseconds.

The mapping between ticks and seconds can be configured.

- Cycle
  
  Cycle is the minimum time unit in the system. The cycle duration is determined by the system clock frequency, that is, the number of cycles per second.
  
- Tick

  Tick is the basic time unit of the operating system and is determined by the number of ticks per second configured by the user.

The OpenHarmony time management module provides time conversion, statistics, and delay functions.


## Development Guidelines

Before you start, learn about the system time and the APIs for time management.


### Available APIs

The following table describes APIs for OpenHarmony LiteOS-A time management. For more details about the APIs, see the API reference.

**Table 1** APIs for time management

| Category| API Description                                                    |
| -------- | ------------------------------------------------------------ |
| Time conversion| **LOS_MS2Tick**: converts milliseconds to ticks.<br>**LOS_Tick2MS**: converts ticks to milliseconds. |
| Time statistics| **LOS_TickCountGet**: obtains the number of current ticks.<br>**LOS_CyclePerTickGet**: obtains the number of cycles of each tick.|


### How to Develop

1. Call APIs to convert time.

2. Call APIs to perform time statistics.

> **NOTE**
>
>  - The system tick count can be obtained only after the system clock is enabled.
>
>  - The time management module depends on **OS_SYS_CLOCK** and **LOSCFG_BASE_CORE_TICK_PER_SECOND** in **los_config.h**.
>
>  - The number of system ticks is not counted when the interrupt feature is disabled. Therefore, the number of ticks cannot be used as the accurate time.


### Development Example

**Prerequisites**

The following parameters are configured: 

- **LOSCFG_BASE_CORE_TICK_PER_SECOND**: number of ticks/second. The value range is (0, 1000).

- **OS_SYS_CLOCK**: system clock, in Hz.

**Sample Code**

Time conversion:

```
VOID Example_TransformTime(VOID)
{
    UINT32 uwMs;
    UINT32 uwTick;
    uwTick = LOS_MS2Tick(10000); // Convert 10000 ms to ticks.
    PRINTK("uwTick = %d \n",uwTick);
    uwMs= LOS_Tick2MS(100); // Convert 100 ticks to ms.
    PRINTK("uwMs = %d \n",uwMs);
}
```

Time statistics and delay:


```
VOID Example_GetTime(VOID)
{
    UINT32 uwcyclePerTick;
    UINT64 uwTickCount;

    uwcyclePerTick = LOS_CyclePerTickGet(); // Obtain the number of cycles per tick.
    if(0 != uwcyclePerTick)
    {
        PRINTK("LOS_CyclePerTickGet = %d \n", uwcyclePerTick);
    }

    uwTickCount = LOS_TickCountGet(); // Obtain the number of ticks.
    if(0 != uwTickCount)
    {
        PRINTK("LOS_TickCountGet = %d \n", (UINT32)uwTickCount);
    }
    LOS_TaskDelay(200);// Delay 200 ticks.
    uwTickCount = LOS_TickCountGet();
    if(0 != uwTickCount)
    {
        PRINTK("LOS_TickCountGet after delay = %d \n", (UINT32)uwTickCount);
    }
}
```

**Verification**

The result is as follows:

Time conversion:


```
uwTick = 10000 
uwMs = 100
```

Time statistics and delay:


```
LOS_CyclePerTickGet = 49500 
LOS_TickCountGet = 347931
LOS_TickCountGet after delay = 348134
```
