# HAR
A Harmony Archive (HAR) is a static shared package that can contain code, C++ libraries, resources, and configuration files. It enables modules and projects to share code related to ArkUI components, resources, and more. Unlike a Harmony Ability Package (HAP), a HAR cannot be independently installed on a device. Instead, it can be referenced only as the dependency of an application module.

## Creating a HAR Module
You can kickstart your HAR module development with the module template of the **Library** type in DevEco Studio. By default, obfuscation is disabled for the HAR module. To enable this feature, set **artifactType** in the **build-profile.json5** file of the HAR module to **obfuscation** as follows:

```json
{
  "apiType": "stageMode",
  "buildOption": {
      "artifactType": "obfuscation"
  }
}
```
The value options of **artifactType** are as follows, and the default value is **original**:
- **original**: Code is not obfuscated.
- **obfuscation**: Code is obfuscated using Uglify.

When obfuscation is enabled, DevEco Studio compiles, obfuscates, and compresses code during HAR building, thereby protecting your code assets.

> **NOTE**
>
> If **artifactType** is set to **obfuscation**, **apiType** must be set to **stageMode**, because obfuscation is available only in the stage model.

## Precautions for HAR Development
- The HAR does not support the declaration of **abilities** and **extensionAbilities** in its configuration file.
- The HAR does not support the declaration of pages in its configuration file.
- The HAR does not support **worker** configuration under **buildOption** in the **build-profile.json5** file.
- The HAR of the FA model and that of the stage model cannot be referenced by each other.
- The HAR of the stage model cannot reference content in the **AppScope** folder. This is because the content in the **AppScope** folder is not packaged into the HAR during compilation and building.

## Exporting ArkUI Components, APIs, and Resources of the HAR
The **index.ets** file acts as the entry of the HAR export declaration file and is where the HAR exports APIs. This file is automatically generated by DevEco Studio by default. You can specify another file as the entry declaration file in the **main** field in the **oh-package.json5** file of the module. The code snippet is as follows:
```json
{
  "main": "index.ets"
}
```
### Exporting ArkUI Components
Use **export** to export the ArkUI components. The code snippet is as follows:
```js
// library/src/main/ets/components/MainPage/MainPage.ets
@Component
export struct MainPage {
  @State message: string = 'Hello World'
  build() {
    Row() {
      Column() {
        Text(this.message)
          .fontSize(50)
          .fontWeight(FontWeight.Bold)
      }
      .width('100%')
    }
    .height('100%')
  }
}
```
In the **index.ets** file, declare the APIs that the HAR exposes to external systems. The code snippet is as follows:
```js
// library/index.ets
export { MainPage } from './src/main/ets/components/MainPage/MainPage'
```
### Exporting TS Classes and Methods
Use **export** to export TS classes and methods. Multiple TS classes and methods can be exported at the same time. The code snippet is as follows:
```js
// library/src/main/ts/test.ets
export class Log {
    static info(msg) {
        console.info(msg);
    }
}

export function func() {
  return "har func";
}

export function func2() {
  return "har func2";
}
```
In the **index.ets** file, declare the APIs that the HAR exposes to external systems. The code snippet is as follows:
```js
// library/index.ets
export { Log } from './src/main/ts/test'
export { func } from './src/main/ts/test'
export { func2 } from './src/main/ts/test'
```
### Resources
Resources are packed into the HAR when it is being compiled and packaged. During compilation and building of a HAP, DevEco Studio collects resource files from the HAP module and its dependent modules. If the resource files of different modules have the same name, DevEco Studio overwrites the resource files based on the following priorities (in descending order):
- AppScope (supported only by the stage model of API version 9)
- Modules in the HAP file
- If resource conflicts occur between dependent HAR modules, they are overwritten based on the dependency sequence. (The module that is higher in the dependency sequence list has higher priority.)

## Referencing ArkUI Components, APIs, and Resources in the HAR
To start with, [configure dependency](https://developer.harmonyos.com/cn/docs/documentation/doc-guides/ohos-development-npm-package-0000001222578434#section89674298391) on the HAR.

### Reference ArkUI Components in the HAR

After configuring the dependency on the HAR, you can reference ArkUI components exported from the HAR by using **import**. The sample code is as follows:
```js
// entry/src/main/ets/pages/index.ets
import { MainPage } from "@ohos/library"

@Entry
@Component
struct Index {
  @State message: string = 'Hello World'
  build() {
    Row() {
      // Reference the ArkUI component in the HAR.
      MainPage()
      Column() {
        Text(this.message)
          .fontSize(50)
          .fontWeight(FontWeight.Bold)
      }
      .width('100%')
    }
    .height('100%')
  }
}
```
### Referencing TS Classes and Methods in the HAR
To reference the TS classes and methods exported from the HAR, use **import** as follows:
```js
// entry/src/main/ets/pages/index.ets
import { Log } from "@ohos/library"
import { func } from "@ohos/library"

@Entry
@Component
struct Index {
  build() {
    Row() {
      Column() {
        Button('Button')
          .onClick(()=>{
            // Reference TS classes and methods in the HAR.
            Log.info("har msg");
            func();
        })
      }
      .width('100%')
    }
    .height('100%')
  }
}
```
### Referencing Resources in the HAR
Use **$r** to reference resources in the HAR. For example, add the **name: hello_har** string (defined in the **string.json** file) and **icon_har.png** image to the **src/main/resources** directory of the HAR module, and then reference the string and image in the entry module. The code snippet is as follows:
```js
// entry/src/main/ets/pages/index.ets
@Entry
@Component
struct Index {
  build() {
    Row() {
      Column() {
        // Reference the string in the HAR.
        Text($r("app.string.hello_har"))
          .fontSize(50)
          .fontWeight(FontWeight.Bold)
        // Reference the image in the HAR.
        Image($r("app.media.icon_har"))
      }
      .width('100%')
    }
    .height('100%')
  }
}
```
