# Audio Recording Development

## When to Use

During audio recording, audio signals are captured, encoded, and saved to files. You can specify parameters such as the sampling rate, number of audio channels, encoding format, encapsulation format, and file path for audio recording.

**Figure 1** Audio recording state transition

![en-us_image_audio_recorder_state_machine](figures/en-us_image_audio_recorder_state_machine.png)



**Figure 2** Layer 0 diagram of audio recording

![en-us_image_audio_recorder_zero](figures/en-us_image_audio_recorder_zero.png)

## How to Develop

For details about the APIs used for audio recording, see [js-apis-media.md](../reference/apis/js-apis-media.md).

### Full-Process Scenario

The full audio recording process includes creating an instance, setting recording parameters, starting, pausing, resuming, and stopping recording, and releasing resources.

```js
import media from '@ohos.multimedia.media'
import mediaLibrary from '@ohos.multimedia.mediaLibrary'

let testFdNumber;

function SetCallBack(audioRecorder) {
    audioRecorder.on('prepare', () => {              								// Set the 'prepare' event callback.
        console.log('prepare success');    
        // The recording page is ready. You can click the Record button to start recording.
    });
    audioRecorder.on('start', () => {    		     								// Set the 'start' event callback.
    	console.log('audio recorder start success');
        // The Record button is changed to the pausable state.
    });
    audioRecorder.on('pause', () => {    		     								// Set the 'pause' event callback.
        console.log('audio recorder pause success');
        // The Record button is changed to the recordable state.
    });
    audioRecorder.on('resume', () => {    		     								// Set the 'resume' event callback.
        console.log('audio recorder resume success');
        // The Record button is changed to the pausable state.
    });
    audioRecorder.on('stop', () => {    		     								// Set the 'stop' event callback.
        console.log('audio recorder stop success');
    });
    audioRecorder.on('release', () => {    		     								// Set the 'release' event callback.
        console.log('audio recorder release success');
    });
    audioRecorder.on('reset', () => {    		     								// Set the 'reset' event callback.
        console.log('audio recorder reset success');
        // You need to reset the recording parameters for another recording.
    });
    audioRecorder.on('error', (error) => {             								// Set the 'error' event callback.
        console.info(`audio error called, errName is ${error.name}`);
        console.info(`audio error called, errCode is ${error.code}`);
        console.info(`audio error called, errMessage is ${error.message}`);
    });
}

// pathName indicates the passed recording file name, for example, 01.mp3. The generated file address is /storage/media/100/local/files/Movies/01.mp3.
// To use the media library, declare the following permissions: ohos.permission.MEDIA_LOCATION, ohos.permission.WRITE_MEDIA, and ohos.permission.READ_MEDIA.
async function getFd(pathName) {
    let displayName = pathName;
    const mediaTest = mediaLibrary.getMediaLibrary();
    let fileKeyObj = mediaLibrary.FileKey;
    let mediaType = mediaLibrary.MediaType.VIDEO;
    let publicPath = await mediaTest.getPublicDirectory(mediaLibrary.DirectoryType.DIR_VIDEO);
    let dataUri = await mediaTest.createAsset(mediaType, displayName, publicPath);
    if (dataUri != undefined) {
        let args = dataUri.id.toString();
        let fetchOp = {
            selections : fileKeyObj.ID + "=?",
            selectionArgs : [args],
        }
        let fetchFileResult = await mediaTest.getFileAssets(fetchOp);
        let fileAsset = await fetchFileResult.getAllObject();
        let fdNumber = await fileAsset[0].open('Rw');
        fdNumber = "fd://" + fdNumber.toString();
        testFdNumber = fdNumber;
    }
}

await getFd('01.mp3');

// 1. Create an AudioRecorder instance.
let audioRecorder = media.createAudioRecorder();    
// 2. Set the callbacks.
SetCallBack(audioRecorder);    
// 3. Set the recording parameters.
let audioRecorderConfig = {
    audioEncoder : media.AudioEncoder.AAC_LC ,
    audioEncodeBitRate : 22050,
    audioSampleRate : 22050,
    numberOfChannels : 2,
    format : media.AudioOutputFormat.AAC_ADTS,
    uri : testFdNumber,                             // testFdNumber is generated by getFd.
    location : { latitude : 30, longitude : 130},
}																					
audioRecorder.prepare(audioRecorderConfig);
// 4. Start recording.
audioRecorder.start();                            	// The start API can be called to trigger the 'start' event callback only after the 'prepare' event callback is complete.
// 5. Pause recording.
audioRecorder.pause();                             	// The pause API can be called to trigger the 'pause' event callback only after the 'start' event callback is complete.
// 6. Resume recording.
audioRecorder.resume();                             // The resume API can be called to trigger the 'resume' event callback only after the 'pause' event callback is complete.
// 7. Stop recording.
audioRecorder.stop();                             	// The stop API can be called to trigger the 'stop' event callback only after the 'start' or 'resume' event callback is complete.
// 8. Reset recording.
audioRecorder.reset();                              // The prepare API can be called for another recording only after the 'reset' event callback is complete.
// 9. Release resources.
audioRecorder.release();                           	// The AudioRecorder resource is destroyed.
audioRecorder = undefined;
```

### Normal Recording Scenario

Unlike the full-process scenario, the normal recording scenario does not include the process of pausing and resuming recording.

```js
import media from '@ohos.multimedia.media'
import mediaLibrary from '@ohos.multimedia.mediaLibrary'

let testFdNumber;

function SetCallBack(audioPlayer) {
    audioRecorder.on('prepare', () => {              								// Set the 'prepare' event callback.
        console.log('prepare success');    
        // The recording page is ready. You can click the Record button to start recording.
    });
    audioRecorder.on('start', () => {    		     								// Set the 'start' event callback.
    	console.log('audio recorder start success');
        // The Record button is changed to the pausable state.
    });  
    audioRecorder.on('stop', () => {    		     								// Set the 'stop' event callback.
        console.log('audio recorder stop success');
    });    
    audioRecorder.on('release', () => {    		     								// Set the 'release' event callback.
        console.log('audio recorder release success');
    });    
}

// pathName indicates the passed recording file name, for example, 01.mp3. The generated file address is /storage/media/100/local/files/Movies/01.mp3.
// To use the media library, declare the following permissions: ohos.permission.MEDIA_LOCATION, ohos.permission.WRITE_MEDIA, and ohos.permission.READ_MEDIA.
async function getFd(pathName) {
    let displayName = pathName;
    const mediaTest = mediaLibrary.getMediaLibrary();
    let fileKeyObj = mediaLibrary.FileKey;
    let mediaType = mediaLibrary.MediaType.VIDEO;
    let publicPath = await mediaTest.getPublicDirectory(mediaLibrary.DirectoryType.DIR_VIDEO);
    let dataUri = await mediaTest.createAsset(mediaType, displayName, publicPath);
    if (dataUri != undefined) {
        let args = dataUri.id.toString();
        let fetchOp = {
            selections : fileKeyObj.ID + "=?",
            selectionArgs : [args],
        }
        let fetchFileResult = await mediaTest.getFileAssets(fetchOp);
        let fileAsset = await fetchFileResult.getAllObject();
        let fdNumber = await fileAsset[0].open('Rw');
        fdNumber = "fd://" + fdNumber.toString();
        testFdNumber = fdNumber;
    }
}

await getFd('01.mp3');

// 1. Create an AudioRecorder instance.
let audioRecorder = media.createAudioRecorder();   
// 2. Set the callbacks.
SetCallBack(audioRecorder);       
// 3. Set the recording parameters.
let audioRecorderConfig = {
    audioEncoder : media.AudioEncoder.AAC_LC ,
    audioEncodeBitRate : 22050,
    audioSampleRate : 22050,
    numberOfChannels : 2,
    format : media.AudioOutputFormat.AAC_ADTS,
    uri : testFdNumber,                             // testFdNumber is generated by getFd.
    location : { latitude : 30, longitude : 130},
}
audioRecorder.prepare(audioRecorderConfig)
// 4. Start recording.
audioRecorder.start();                            	// The start API can be called to trigger the 'start' event callback only after the 'prepare' event callback is complete.
// 5. Stop recording.
audioRecorder.stop();                             	// The stop API can be called to trigger the 'stop' event callback only after the 'start' or 'resume' event callback is complete.
// 6. Release resources.
audioRecorder.release();                           	// The AudioRecorder resource is destroyed.
audioRecorder = undefined;
```
