# Data Ability Development

## Basic Concepts
A Data ability helps applications manage access to data stored by themselves and other applications. It also provides APIs for sharing data with other applications either on the same device or across devices.

Data ability providers can customize data access-related APIs such as data inserting, deleting, updating, and querying, as well as file opening, and share data with other applications through these open APIs.

## Creating a Data Ability
### 1. Implementation of the Data Subsystem
1. To meet the basic requirements of the database storage service, implement the **Insert**, **Query**, **Update**, and **Delete** APIs in the **Data** class to provide batch data processing. The traversal logic has been implemented by the **BatchInsert** and **ExecuteBatch** APIs.
2. The APIs used in the lifecycle of the Data ability are as follows:

   - onInitialized

     Called during ability initialization to initialize the relational database (RDB).

   - update

     Updates data in the database.

   - query

     Queries data in the database.

   - delete

     Deletes one or multiple data records from the database.

   - normalizeUri

     Normalizes the URI. A normalized URI applies to cross-device use, persistence, backup, and restore. When the context changes, it ensures that the same data item can be referenced.

   - batchInsert

     Inserts multiple data records into the database.

   - denormalizeUri

     Converts a normalized URI generated by **normalizeUri** into a denormalized URI.

   - insert

     Inserts a data record into the database.

   - openFile

     Opens a file.

   - getFileTypes

     Obtains the MIME type of a file.

   - getType

     Obtains the MIME type matching the data specified by the URI.

   - executeBatch
   
     Operates data in the database in batches.

   - call

     A customized API.
 

   The following code snippet shows how to create a Data ability:

   ```javascript
    import dataAbility from '@ohos.data.dataability'
    import dataRdb from '@ohos.data.rdb'
    
    const TABLE_NAME = 'book'
    const STORE_CONFIG = { name: 'book.db', encryptKey: new Uint8Array([]) }
    const SQL_CREATE_TABLE = 'CREATE TABLE IF NOT EXISTS book(id INTEGER PRIMARY KEY AUTOINCREMENT, name TEXT NOT NULL, age INTEGER, introduction TEXT NOT NULL)'
    let rdbStore =  undefined
    
    export default {
        onInitialized(abilityInfo) {
            console.info('DataAbility onInitialized, abilityInfo:' + abilityInfo.bundleName)
            dataRdb.getRdbStore(STORE_CONFIG, 1, (err, store) => {
                console.info('DataAbility getRdbStore callback')
                store.executeSql(SQL_CREATE_TABLE, [])
                rdbStore = store
            });
        },
        insert(uri, valueBucket, callback) {
            console.info('DataAbility insert start')
            rdbStore.insert(TABLE_NAME, valueBucket, callback)
        },
        batchInsert(uri, valueBuckets, callback) {
            console.info('DataAbility batch insert start')
            for (let i = 0;i < valueBuckets.length; i++) {
                console.info('DataAbility batch insert i=' + i)
                if (i < valueBuckets.length - 1) {
                    rdbStore.insert(TABLE_NAME, valueBuckets[i], (num: number) => {
                        console.info('DataAbility batch insert ret=' + num)
                    })
                } else {
                    rdbStore.insert(TABLE_NAME, valueBuckets[i], callback)
                }
            }
        },
        query(uri, columns, predicates, callback) {
            console.info('DataAbility query start')
            let rdbPredicates = dataAbility.createRdbPredicates(TABLE_NAME, predicates)
            rdbStore.query(rdbPredicates, columns, callback)
        },
        update(uri, valueBucket, predicates, callback) {
            console.info('DataAbilityupdate start')
            let rdbPredicates = dataAbility.createRdbPredicates(TABLE_NAME, predicates)
            rdbStore.update(valueBucket, rdbPredicates, callback)
        },
        delete(uri, predicates, callback) {
            console.info('DataAbilitydelete start')
            let rdbPredicates = dataAbility.createRdbPredicates(TABLE_NAME, predicates)
            rdbStore.delete(rdbPredicates, callback)
        }
    };

   ```


### 2. Subsystem Configuration

| JSON Field | Description                                                    |
| ------------- | ------------------------------------------------------------ |
| "name"        | Ability name, corresponding to the **Data** class name derived from **Ability**.                       |
| "type"        | Ability type, which is **Data** for a Data ability.                   |
| "uri"         | URI used for communication.                                               |
| "visible"     | Whether the Data ability is visible to other applications. When this parameter is set to **true**, the Data ability can communicate with other applications.|

**config.json configuration example**

```json
"abilities":[{
    "srcPath": "DataAbility",
    "name": ".DataAbility",
    "icon": "$media:icon",
    "srcLanguage": "ets",
    "description": "$string:description_dataability",
    "type": "data",
    "visible": true,
    "uri": "dataability://ohos.samples.etsdataability.DataAbility"    
}]
```

## Accessing a Data ability
### 1. Preparing for JS Application Development
Basic dependency packages:
    1. @ohos.ability.featureAbility
    2. @ohos.data.dataability
    3. @ohos.data.rdb
URI string used for communication with the Data ability.

### 2. JS Application Development APIs
Create a utility API object.
```js
// Different from the URI defined in the config.json file, the URI passed in the parameter has an extra slash (/), because there is a DeviceID parameter between the second and the third slash (/).
var urivar = "dataability:///com.ix.DataAbility"
var DAHelper = featureAbility.acquireDataAbilityHelper(
    urivar
);
```
Construct RDB data.
```js
var valuesBucket = {"name": "gaolu"}
var da = new ohos_data_ability.DataAbilityPredicates()
var valArray =new Array("value1");
var cars = new Array({"batchInsert1" : "value1",});
```
Use **insert** to insert data to the Data subsystem.
```js
// Callback mode:
DAHelper.insert(
    urivar,
    valuesBucket,
    (error, data) => {
        expect(typeof(data)).assertEqual("number")
    }
);
// Promise mode:
var datainsert = await DAHelper.insert(
    urivar,
    valuesBucket
);
```
Use **delete** to delete data from the Data subsystem.
```js
// Callback mode:
DAHelper.delete(
    urivar,
    da,
    (error, data) => {
        expect(typeof(data)).assertEqual("number")
    }
);
// Promise mode:
var datadelete = await DAHelper.delete(
    urivar,
    da,
);
```
Use **update** to update data in the Data subsystem.
```js
// Callback mode:
DAHelper.update(
    urivar
    valuesBucket,
    da,
    (error, data) => {
        expect(typeof(data)).assertEqual("number")
    }
);
// Promise mode:
var dataupdate = await DAHelper.update(
    urivar,
    valuesBucket,
    da,
);
```
Use **query** to query data in the Data subsystem.
```js
// Callback mode:
DAHelper.query(
    urivar,
    valArray,
    da,
    (error, data) => {
        expect(typeof(data)).assertEqual("object")
    }
);
// Promise mode:
var dataquery = await DAHelper.query(
    urivar,
    valArray,
    da
);
```
Use **batchInsert** to insert data in batches to the Data subsystem.
```js
// Callback mode:
DAHelper.batchInsert(
    urivar,
    cars,
    (error, data) => {
        expect(typeof(data)).assertEqual("number")
    }
);
// Promise mode:
var databatchInsert = await DAHelper.batchInsert(
    urivar,
    cars
);
```
Use **executeBatch** to process data in batches in the Data subsystem.
```js
// Callback mode:
DAHelper.executeBatch(
    urivar,
    [
        {
            uri: urivar,
            type: featureAbility.DataAbilityOperationType.TYPE_INSERT,
            valuesBucket: {"executeBatch" : "value1",},
            predicates: da,
            expectedCount:0,
            PredicatesBackReferences: {},
            interrupted:true,
        }
    ],
    (error, data) => {
        expect(typeof(data)).assertEqual("object")
    }
);
// Promise mode:
var dataexecuteBatch = await DAHelper.executeBatch(
    urivar,
    [
        {
            uri: urivar,
            type: featureAbility.DataAbilityOperationType.TYPE_INSERT,
            valuesBucket:
            {
                "executeBatch" : "value1",
            },
            predicates: da,
            expectedCount:0,
            PredicatesBackReferences: {},
            interrupted:true,
        }
    ]
);
```

## Development Example

The following sample is provided to help you better understand how to develop a Data ability:

- [eTSDataAbility](https://gitee.com/openharmony/app_samples/tree/master/ability/eTSDataAbility)

This **eTSDataAbility** sample shows how to:

Create a Data ability in the **data.ts** file in the **DataAbility** directory.

Encapsulate the process of accessing the Data ability in the **MainAbility** directory.
