// SPDX-License-Identifier: GPL-2.0+
// Copyright (c) 2016-2017 Hisilicon Limited.

#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/netdevice.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <net/sock.h>
#include <net/rtnetlink.h>

#include "hns3_enet.h"
#include "hclge_cmd.h"
#include "hclge_main.h"
#include "hns3_nictool.h"
#include "hns3_priv_tm.h"
#include "hns3_priv_dcb.h"
#include "hns3_priv_pkt.h"
#include "hns3_priv_mac.h"
#include "hns3_priv_dfx.h"
#include "hns3_priv_vlan.h"
#include "hns3_priv_phy.h"
#include "hns3_priv_qos.h"
#include "hns3_priv_qinfo.h"
#include "hns3_priv_promisc.h"
#include "hns3_priv_fd.h"
#include "hns3_priv_rss.h"
#include "hns3_priv_common_test.h"
#include "hns3_priv_m7_cmd.h"
#include "hns3_priv_qres.h"
#include "hns3_priv_stat.h"
#include "hns3_priv_irq.h"
#include "hns3_priv_lamp.h"
#include "hns3_priv_ext.h"
#include "hns3_priv_xsfp.h"
#include "hns3_priv_port.h"
#include "hns3_priv_hilink_param.h"
#include "hns3_priv_mactbl.h"
#include "kcompat.h"

#define HCLGE_CHS_OUT_L3_B 0
#define HCLGE_CHS_OUT_UDP_B 1
#define HCLGE_CHS_INNER_L3_B 0
#define HCLGE_CHS_INNER_TCP_B 1
#define HCLGE_CHS_INNER_UDP_B 2
#define HCLGE_CHS_INNER_SCTP_B 3
#define HCLGE_OPC_CHECKSUM_CHECK_EN 0x0601
#define SCC_TEMP_LOW_ADDR	0x31000000
#define SCC_TEMP_HIGH_ADDR	0x1

/* misc command */
#define HCLGE_OPC_CHIP_ID_GET		0x7003
#define HCLGE_OPC_IMP_COMMIT_ID_GET  0x7004
#define HCLGE_OPC_GET_CHIP_NUM		0x7005
#define HCLGE_OPC_GET_PORT_NUM		0x7006
/* SFP command */
#define HCLGE_OPC_SFP_GET_INFO		0x7100
#define HCLGE_OPC_SFP_GET_PRESENT	0x7101
#define HCLGE_OPC_SFP_SET_STATUS	0x7102
/* DCQCN command */
#define HCLGE_OPC_DCQCN_TEMPLATE_CFG	0x7014
#define HCLGE_OPC_DCQCN_GET_MSG_CNT		0x7017

#define HNAE_DRIVER_VERSION "1.8.10.1"

#define MAX_MSG_OUT_SIZE	(1024U * 2048U)
#define MAX_MSG_IN_SIZE		(1024U * 2048U)

struct hclge_chs_param {
	u8 outer_en;
	u8 inner_en;
	u8 rsv[22];
};

struct tx_timeout_param {
	u16 wr_flag;
	u16 tx_timeout_size;
};

struct reset_param {
	u32 reset_level;
};

static dev_t g_dev_id = {0};

struct class *g_nictool_class;
struct cdev g_nictool_cdev;
static const char hns3_driver_name[] = "hns3";

int g_nictool_init_flag;
int g_nictool_ref_cnt;

typedef int (*driv_module) (struct hns3_nic_priv *nic_dev, void *buf_in,
			    u16 in_size, void *buf_out, u16 *out_size);

struct drv_module_handle {
	enum driver_cmd_type driv_cmd_name;
	driv_module driv_func;
};

static void free_buff_in(void *buf_in)
{
	if (!buf_in)
		return;

	kfree(buf_in);
}

static int alloc_buff_in(struct msg_module *nt_msg, u32 in_size, void **buf_in)
{
	void *msg_buf;

	if (!in_size)
		return 0;

	if (in_size > MAX_MSG_IN_SIZE) {
		pr_err("msg in size(%u) more than %u\n",
		       in_size, MAX_MSG_IN_SIZE);
		return -ENOMEM;
	}

	msg_buf = kzalloc((unsigned long)in_size, GFP_KERNEL);
	*buf_in = msg_buf;
	if (ZERO_OR_NULL_PTR(*buf_in)) {
		pr_err("alloc buf_in failed\n");
		return -ENOMEM;
	}

	if (copy_from_user(msg_buf, nt_msg->in_buff, (unsigned long)in_size)) {
		pr_err("Copy from user failed in %s function\n", __func__);
		kfree(msg_buf);
		return -EFAULT;
	}

	return 0;
}

static void free_buff_out(void *buf_out)
{
	if (!buf_out)
		return;

	kfree(buf_out);
}

static int alloc_buff_out(u32 out_size, void **buf_out)
{
	if (!out_size)
		return 0;

	if (out_size > MAX_MSG_OUT_SIZE) {
		pr_err("msg out size(%u) more than %u\n",
		       out_size, MAX_MSG_OUT_SIZE);
		return -ENOMEM;
	}

	*buf_out = kzalloc((unsigned long)out_size, GFP_KERNEL);
	if (ZERO_OR_NULL_PTR(*buf_out)) {
		pr_err("alloc buf_out failed\n");
		return -ENOMEM;
	}

	return 0;
}

static int copy_buf_out_to_user(struct msg_module *nt_msg, u32 out_size,
				void **buf_out)
{
	int ret = 0;
	void *msg_out = buf_out;

	if (copy_to_user(nt_msg->out_buf, msg_out, out_size))
		ret = -EFAULT;

	return ret;
}

static int nictool_netdev_match_check(struct net_device *netdev)
{
	struct ethtool_drvinfo drv_info;

	if (netdev->ethtool_ops->get_drvinfo)
		netdev->ethtool_ops->get_drvinfo(netdev, &drv_info);

	if (!strncmp(drv_info.driver, hns3_driver_name,
		     strlen(hns3_driver_name)))
		return 0;

	netdev_err(netdev, "match hns3 driver name(%s) failed\n",
		   drv_info.driver);
	return -1;
}

#if (KERNEL_VERSION(4, 16, 0) < LINUX_VERSION_CODE)
static int kernel_sock_ioctl(struct socket *sock, int cmd, unsigned long arg)
{
	mm_segment_t oldfs = get_fs();
	int err;

	set_fs(KERNEL_DS);
	err = sock->ops->ioctl(sock, cmd, arg);
	set_fs(oldfs);

	return err;
}
#endif

static struct net_device *get_netdev_by_ifname(char *ifname)
{
	struct socket *temp_sock = NULL;
	struct net_device *netdev = NULL;
	struct ifreq ifr;
	int err;

	err = sock_create(PF_INET, SOCK_DGRAM, 0, &temp_sock);
	if (err < 0) {
		pr_err("fail to enter sock_create, err = %d\n", err);
		return NULL;
	}

	strncpy(ifr.ifr_ifrn.ifrn_name, ifname, (unsigned long)IFNAMSIZ);
	kernel_sock_ioctl(temp_sock, SIOCSIFNAME, (unsigned long)&ifr);
	netdev = dev_get_by_name(sock_net(temp_sock->sk), ifname);
	if (!netdev)
		goto out;

	dev_put(netdev);

out:
	sock_release(temp_sock);
	return netdev;
}

static int nictool_k_get_netdev_by_ifname(char *ifname,
					  struct hns3_nic_priv **nic_dev)
{
	struct net_device *netdev = NULL;

	netdev = get_netdev_by_ifname(ifname);
	if (!netdev) {
		pr_err("not find the netdevice(%s)!\n", ifname);
		return -EFAULT;
	}

	if (nictool_netdev_match_check(netdev)) {
		netdev_err(netdev, "netdevice is not hns device.\n");
		return -EFAULT;
	}

	*nic_dev = (struct hns3_nic_priv *)netdev_priv(netdev);
	if (!(*nic_dev)) {
		netdev_err(netdev, "no private data\n");
		return -EFAULT;
	}

	return 0;
}

int hns3_test_chs_set(struct hclge_dev *hdev, u8 chs_type, u8 enable)
{
	struct hclge_desc desc;
	enum hclge_cmd_status status;
	struct hclge_chs_param *recv;

	hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_CHECKSUM_CHECK_EN, true);
	status = hclge_cmd_send(&hdev->hw, &desc, 1);
	if (status) {
		pr_err("chs get cmd send failed!\n");
		return status;
	}

	recv = (struct hclge_chs_param *)desc.data;

	switch (chs_type) {
	case CKS_OUTER_L3_EN:
		hnae3_set_bit(recv->outer_en, HCLGE_CHS_OUT_L3_B, enable);
		break;
	case CKS_OUTER_UDP_EN:
		hnae3_set_bit(recv->outer_en, HCLGE_CHS_OUT_UDP_B, enable);
		break;
	case CKS_INNER_L3_EN:
		hnae3_set_bit(recv->inner_en, HCLGE_CHS_INNER_L3_B, enable);
		break;
	case CKS_INNER_TCP_EN:
		hnae3_set_bit(recv->inner_en, HCLGE_CHS_INNER_TCP_B, enable);
		break;
	case CKS_INNER_UDP_EN:
		hnae3_set_bit(recv->inner_en, HCLGE_CHS_INNER_UDP_B, enable);
		break;
	case CKS_INNER_SCTP_EN:
		hnae3_set_bit(recv->inner_en, HCLGE_CHS_INNER_SCTP_B, enable);
		break;
	default:
		break;
	}

	hclge_cmd_reuse_desc(&desc, false);
	status = hclge_cmd_send(&hdev->hw, &desc, 1);

	return status;
}

int hns3_test_chs_get(struct hclge_dev *hdev, u8 chs_type, u8 *enable)
{
	struct hclge_chs_param *recv;
	enum hclge_cmd_status status;
	struct hclge_desc desc;
	u8 inner_sctp_en;
	u8 inner_tcp_en;
	u8 inner_udp_en;
	u8 outer_udp_en;
	u8 outer_l3_en;
	u8 inner_l3_en;

	hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_CHECKSUM_CHECK_EN, true);
	status = hclge_cmd_send(&hdev->hw, &desc, 1);
	if (status) {
		pr_err("chs get cmd send failed!\n");
		return status;
	}

	recv = (struct hclge_chs_param *)desc.data;
	outer_l3_en = hnae3_get_bit(recv->outer_en, HCLGE_CHS_OUT_L3_B);
	outer_udp_en = hnae3_get_bit(recv->outer_en, HCLGE_CHS_OUT_UDP_B);
	inner_l3_en = hnae3_get_bit(recv->inner_en, HCLGE_CHS_INNER_L3_B);
	inner_tcp_en = hnae3_get_bit(recv->inner_en, HCLGE_CHS_INNER_TCP_B);
	inner_udp_en = hnae3_get_bit(recv->inner_en, HCLGE_CHS_INNER_UDP_B);
	inner_sctp_en = hnae3_get_bit(recv->inner_en, HCLGE_CHS_INNER_SCTP_B);

	switch (chs_type) {
	case CKS_OUTER_L3_EN:
		*enable = outer_l3_en;
		break;
	case CKS_OUTER_UDP_EN:
		*enable = outer_udp_en;
		break;
	case CKS_INNER_L3_EN:
		*enable = inner_l3_en;
		break;
	case CKS_INNER_TCP_EN:
		*enable = inner_tcp_en;
		break;
	case CKS_INNER_UDP_EN:
		*enable = inner_udp_en;
		break;
	case CKS_INNER_SCTP_EN:
		*enable = inner_sctp_en;
		break;
	default:
		break;
	}

	return status;
}

int hns3_test_chs_cfg(struct hns3_nic_priv *net_priv,
		      void *buf_in, u16 in_size, void *buf_out, u16 *out_size)
{
	struct hns3_chs_param *in_info;
	struct hnae3_handle *handle;
	struct hclge_vport *vport;
	struct hclge_dev *hdev;
	u8 *out_info;
	u8 is_set;

	handle = net_priv->ae_handle;
	vport = hclge_get_vport(handle);
	hdev = vport->back;
	in_info = (struct hns3_chs_param *)buf_in;
	out_info = (u8 *)buf_out;
	is_set = in_info->is_set;

	if (in_info->type >= CKS_MAX) {
		pr_err("chs type is %d, param err!\n", in_info->type);
		return -1;
	}

	if (in_info->is_enable != 0 && in_info->is_enable != 1) {
		pr_err("chs enable is %d, param err!\n", in_info->is_enable);
		return -1;
	}
	if (is_set) {
		if (hns3_test_chs_set(hdev, in_info->type,
				      in_info->is_enable)) {
			pr_err("set chs type(%d) enable failed!\n",
			       in_info->type);
			return -1;
		}
	} else {
		if (hns3_test_chs_get(hdev, in_info->type, out_info)) {
			pr_err("get chs type(%d) enable failed!\n",
			       in_info->type);
			return -1;
		}
		pr_err("chs type(%d) enable status is %d\n",
		       in_info->type, *out_info);
	}

	return 0;
}

int hns_test_get_commit_id(struct hnae3_handle *handle, u8 *commit_id,
			   u32 *ncl_version)
{
#define COMMIT_ID_LEN	8
	struct hclge_vport *vport = hclge_get_vport(handle);
	struct hns3_test_commit_id_param *resp;
	struct hclge_dev *hdev = vport->back;
	struct hclge_desc desc;
	int ret, i;

	hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_IMP_COMMIT_ID_GET, true);
	resp = (struct hns3_test_commit_id_param *)(desc.data);
	ret = hclge_cmd_send(&hdev->hw, &desc, 1);
	if (ret) {
		dev_err(&hdev->pdev->dev, "get commit id failed %d\n", ret);
		return ret;
	}

	for (i = 0; i < COMMIT_ID_LEN; i++)
		commit_id[i] = resp->commit_id[i];

	commit_id[COMMIT_ID_LEN] = '\0';
	*ncl_version = resp->ncl_version;

	return 0;
}

static int get_fw_ver(struct hns3_nic_priv *nic_dev, void *buf_in, u16 in_size,
		      void *buf_out, u16 *out_size)
{
	struct firmware_ver_param *out_buf;
	struct hnae3_handle *handle;
	struct hclge_vport *vport;
	struct hclge_dev *hdev;
	u32 fw_ver;

	handle = nic_dev->ae_handle;
	vport = container_of(handle, struct hclge_vport, nic);
	hdev = vport->back;
	out_buf = (struct firmware_ver_param *)buf_out;

	if (!handle || !hdev)
		return -EFAULT;

	if (hns_test_get_commit_id(handle, out_buf->commit_id,
				   &out_buf->ncl_version))
		return -EFAULT;

	fw_ver = hdev->fw_version;
	out_buf->imp_ver = fw_ver;

	if (!fw_ver)
		return -EFAULT;

	return 0;
}

static int get_driver_ver(struct hns3_nic_priv *nic_dev,
			  void *buf_in, u16 in_size,
			  void *buf_out, u16 *out_size)
{
	if (!buf_out)
		return -ENOMEM;

	strncpy(buf_out, HNAE_DRIVER_VERSION, sizeof(HNAE_DRIVER_VERSION));
	*out_size = sizeof(HNAE_DRIVER_VERSION);

	return 0;
}

int hns3_test_clean_stats(struct hns3_nic_priv *net_priv,
			  void *buf_in, u16 in_size,
			  void *buf_out, u16 *out_size)
{
	struct net_device *netdev = net_priv->netdev;
	struct hnae3_knic_private_info *kinfo;
	struct hnae3_handle *handle;
	struct hns3_enet_ring *ring;
	struct hclge_vport *vport;
	struct hclge_dev *hdev;
	struct hclge_tqp *tqp;
	int i;

	handle = net_priv->ae_handle;
	kinfo = &handle->kinfo;
	vport = container_of(handle, struct hclge_vport, nic);
	hdev = vport->back;

	for (i = 0; i < kinfo->num_tqps; i++) {
		tqp = container_of(kinfo->tqp[i], struct hclge_tqp, q);
		memset(&tqp->tqp_stats, 0, sizeof(struct hlcge_tqp_stats));

		ring = net_priv->ring_data[i].ring;
		memset(&ring->stats, 0, sizeof(struct ring_stats));
		ring = net_priv->ring_data[i + kinfo->num_tqps].ring;
		memset(&ring->stats, 0, sizeof(struct ring_stats));
	}
	memset(&hdev->hw_stats.mac_stats, 0, sizeof(struct hclge_mac_stats));
	memset(&netdev->stats, 0, sizeof(struct net_device_stats));

	return 0;
}

int hns3_nic_reset(struct hns3_nic_priv *net_priv,
		   void *buf_in, u16 in_size, void *buf_out, u16 *out_size)
{
	struct hnae3_handle *h = net_priv->ae_handle;
	struct reset_param *reset_info;
	enum hnae3_reset_type rst_type;
	struct hclge_vport *vport;
	struct hclge_dev *hdev;

	vport = container_of(h, struct hclge_vport, nic);
	hdev = vport->back;
	reset_info = (struct reset_param *)buf_in;
	rst_type = HNAE3_NONE_RESET;

	if (test_bit(HCLGE_STATE_REMOVING, &hdev->state)) {
		dev_info(&hdev->pdev->dev, "driver already uninit!\n");
		return 0;
	}

	if (time_before(jiffies, (hdev->last_reset_time + 12 * HZ)))
		return 0;

	if (reset_info->reset_level == HNAE3_FUNC_RESET)
		rst_type = HNAE3_FUNC_RESET;
	else if (reset_info->reset_level == HNAE3_GLOBAL_RESET)
		rst_type = HNAE3_GLOBAL_RESET;

	hdev->reset_level = rst_type;
	dev_info(&hdev->pdev->dev,
		 "user received reset event, reset type is %d\n",
		 hdev->reset_level);

	/* request reset & schedule reset task */
	set_bit(hdev->reset_level, &hdev->reset_request);
	if (!test_and_set_bit(HCLGE_STATE_RST_SERVICE_SCHED, &hdev->state))
		schedule_work(&hdev->rst_service_task);

	return 0;
}

int hns3_nic_timeout_cfg(struct hns3_nic_priv *net_priv,
			 void *buf_in, u16 in_size,
			 void *buf_out, u16 *out_size)
{
	struct net_device *netdev = net_priv->netdev;
	struct tx_timeout_param *in_info;
	struct tx_timeout_param *out_info;

	in_info = (struct tx_timeout_param *)buf_in;
	out_info = (struct tx_timeout_param *)buf_out;

	if (in_info->wr_flag)
		netdev->watchdog_timeo = (in_info->tx_timeout_size) * HZ;
	else
		out_info->tx_timeout_size = (netdev->watchdog_timeo) / HZ;

	return 0;
}

int hns3_gro_age_handle(struct hns3_nic_priv *net_priv,
			void *buf_in, u16 in_size,
			void *buf_out, u16 *out_size)
{
	struct hnae3_handle *h = net_priv->ae_handle;
	struct hclge_gro_age_config_cmd *req;
	struct hclge_desc desc;
	struct hclge_vport *vport;
	struct hclge_dev *hdev;
	struct gro_param *param;
	int ret;

	vport = container_of(h, struct hclge_vport, nic);
	param = (struct gro_param *)buf_in;
	hdev = vport->back;

	hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_GRO_AGE_CFG,
				   param->is_read ? true : false);
	req = (struct hclge_gro_age_config_cmd *)desc.data;

	if (!param->is_read)
		req->ppu_gro_age_cnt = param->age_cnt;

	ret = hclge_cmd_send(&hdev->hw, &desc, 1);
	if (ret) {
		dev_err(&hdev->pdev->dev, "gro age config fail, ret = %d\n",
			ret);
		return ret;
	}

	if (param->is_read) {
		memcpy(buf_out, &req->ppu_gro_age_cnt,
		       sizeof(req->ppu_gro_age_cnt));
		*out_size = sizeof(req->ppu_gro_age_cnt);
	}

	return 0;
}

static int hns3_dcqcn_rw(struct hns3_nic_priv *net_priv,
			 u32 offset, u32 *data, u32 rw_type)
{
	struct hnae3_handle *h = net_priv->ae_handle;
	struct hclge_vport *vport;
	struct hclge_dev *hdev;
	struct hclge_desc desc;
	int ret;

	if (!data)
		return -EFAULT;

	vport = container_of(h, struct hclge_vport, nic);
	hdev = vport->back;

	if (rw_type == DEVMEM_CFG_READ) {
		hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_DCQCN_TEMPLATE_CFG,
					   true);
	} else {
		hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_DCQCN_TEMPLATE_CFG,
					   false);
		desc.data[2] = *data;
	}

	desc.data[0] = SCC_TEMP_LOW_ADDR + offset;
	desc.data[1] = SCC_TEMP_HIGH_ADDR;
	desc.data[4] = 32;

	ret = hclge_cmd_send(&hdev->hw, &desc, 1);
	if (ret) {
		dev_err(&hdev->pdev->dev, "disable net lane failed %d\n", ret);
		return ret;
	}

	if (rw_type == DEVMEM_CFG_READ)
		*data = desc.data[2];

	return 0;
}

int hns3_nic_dcqcn(struct hns3_nic_priv *net_priv,
		   void *buf_in, u16 in_size, void *buf_out, u16 *out_size)
{
	struct hnae3_handle *h = net_priv->ae_handle;
	struct hclge_vport *vport = container_of(h, struct hclge_vport, nic);
	struct cfg_dcqcn_param *parm_out = buf_out;
	struct cfg_dcqcn_param *parm_in = buf_in;
	struct cfg_dcqcn_param tempbuffer = {0};
	struct hclge_dev *hdev = vport->back;
	u32 tempoutbuff;
	u32 offset;
	int ret;

	if (!hnae3_dev_roce_supported(hdev)) {
		dev_err(&hdev->pdev->dev, "This device is not support RoCE!\n");
		return -EINVAL;
	}

	tempoutbuff = 0;
	if (parm_in->device_number > 0xff) {
		dev_err(&hdev->pdev->dev,
			"parm_in->device_number=0x%x, max value is 0xff.\n",
			parm_in->device_number);
		return -ENXIO;
	}
	offset = 0x10 * parm_in->device_number + 0x6000;
	ret = hns3_dcqcn_rw(net_priv, offset, (u32 *)&tempoutbuff,
			    DEVMEM_CFG_READ);
	if (ret) {
		dev_err(&hdev->pdev->dev,
			"read dcqcn cfg 0~31 bit failed 0x%x\n", ret);
		return ret;
	}
	tempbuffer.ai = (tempoutbuff & 0xffff);
	tempbuffer.f = ((tempoutbuff >> 16) & 0xff);
	tempbuffer.tkp = (tempoutbuff >> 24);

	offset = offset + 0x4;
	ret = hns3_dcqcn_rw(net_priv, offset, (u32 *)&tempoutbuff,
			    DEVMEM_CFG_READ);
	if (ret) {
		dev_err(&hdev->pdev->dev,
			"read dcqcn cfg 32~63 bit failed ret = 0x%x\n", ret);
		return ret;
	}
	tempbuffer.tmp = (tempoutbuff & 0xffff);
	tempbuffer.alp = (tempoutbuff >> 16);

	offset = offset + 0x4;
	ret = hns3_dcqcn_rw(net_priv, offset, (u32 *)&tempoutbuff,
			    DEVMEM_CFG_READ);
	if (ret) {
		dev_err(&hdev->pdev->dev,
			"read dcqcn cfg 64~95 bit failed ret = 0x%x\n", ret);
		return ret;
	}
	tempbuffer.max_speed = tempoutbuff;

	offset = offset + 0x4;
	ret = hns3_dcqcn_rw(net_priv, offset, (u32 *)&tempoutbuff,
			    DEVMEM_CFG_READ);
	if (ret) {
		dev_err(&hdev->pdev->dev,
			"read dcqcn cfg 96~127 bit failed ret = 0x%x\n", ret);
		return ret;
	}
	tempbuffer.g = (tempoutbuff & 0xff);
	tempbuffer.al = ((tempoutbuff >> 8) & 0xff);
	tempbuffer.cnp_time = ((tempoutbuff >> 16) & 0xff);
	tempbuffer.alp_shift = ((tempoutbuff >> 24) & 0xff);

	if (parm_in->is_get == HINICADM_DCQCN_WRITE_CFG_MODE) {
		if ((parm_in->dcqcn_parm_opcode & 0x1) == 1)
			tempbuffer.ai = parm_in->ai;
		if ((parm_in->dcqcn_parm_opcode & 0x2) == 0x2)
			tempbuffer.f = parm_in->f;
		if ((parm_in->dcqcn_parm_opcode & 0x4) == 0x4)
			tempbuffer.tkp = parm_in->tkp;
		if ((parm_in->dcqcn_parm_opcode & 0x8) == 0x8)
			tempbuffer.tmp = parm_in->tmp;
		if ((parm_in->dcqcn_parm_opcode & 0x10) == 0x10)
			tempbuffer.alp = parm_in->alp;
		if ((parm_in->dcqcn_parm_opcode & 0x20) == 0x20)
			tempbuffer.g = parm_in->g;
		if ((parm_in->dcqcn_parm_opcode & 0x40) == 0x40)
			tempbuffer.al = parm_in->al;
		if ((parm_in->dcqcn_parm_opcode & 0x80) == 0x80)
			tempbuffer.max_speed = parm_in->max_speed;
		if ((parm_in->dcqcn_parm_opcode & 0x100) == 0x100)
			tempbuffer.cnp_time = parm_in->cnp_time;
		if ((parm_in->dcqcn_parm_opcode & 0x200) == 0x200)
			tempbuffer.alp_shift = parm_in->alp_shift;

		ret = hns3_dcqcn_rw(net_priv,
				    0x10 * parm_in->device_number + 0x6000,
				    (u32 *)&tempbuffer.ai, DEVMEM_CFG_WRITE);
		if (ret) {
			dev_err(&hdev->pdev->dev,
				"write dcqcn cfg 0~31 bit failed ret = 0x%x\n",
				ret);
			return ret;
		}
		ret = hns3_dcqcn_rw(net_priv,
				    0x10 * parm_in->device_number + 0x6004,
				    (u32 *)&tempbuffer.tmp, DEVMEM_CFG_WRITE);
		if (ret) {
			dev_err(&hdev->pdev->dev,
				"write dcqcn cfg 32~63 bit failed ret = 0x%x\n",
				ret);
			return ret;
		}
		ret = hns3_dcqcn_rw(net_priv,
				    0x10 * parm_in->device_number + 0x6008,
				    (u32 *)&tempbuffer.max_speed,
				    DEVMEM_CFG_WRITE);
		if (ret) {
			dev_err(&hdev->pdev->dev,
				"write dcqcn cfg 64~95 bit failed ret = 0x%x\n",
				ret);
			return ret;
		}
		ret = hns3_dcqcn_rw(net_priv,
				    0x10 * parm_in->device_number + 0x600c,
				    (u32 *)&tempbuffer.g, DEVMEM_CFG_WRITE);
		if (ret) {
			dev_err(&hdev->pdev->dev,
				"write dcqcn cfg 96~127 bit failed ret = 0x%x\n",
				ret);
			return ret;
		}
	} else if (parm_in->is_get == HINICADM_DCQCN_READ_CFG_MODE) {
		parm_out->ai = tempbuffer.ai;
		parm_out->f = tempbuffer.f;
		parm_out->tkp = tempbuffer.tkp;
		parm_out->tmp = tempbuffer.tmp;
		parm_out->alp = tempbuffer.alp;
		parm_out->max_speed = tempbuffer.max_speed;
		parm_out->g = tempbuffer.g;
		parm_out->al = tempbuffer.al;
		parm_out->cnp_time = tempbuffer.cnp_time;
		parm_out->alp_shift = tempbuffer.alp_shift;
	} else {
		dev_err(&hdev->pdev->dev,
			"parm->is_get = 0x%x parm is error type\n",
			parm_in->is_get);
	}

	return 0;
}

int hns3_dcqcn_get_msg_cnt(struct hns3_nic_priv *net_priv,
			   void *buf_in, u16 in_size,
			   void *buf_out, u16 *out_size)
{
	struct hnae3_handle *h = net_priv->ae_handle;
	struct hclge_vport *vport = container_of(h, struct hclge_vport, nic);
	struct dcqcn_statistic_param *statistic_parm_out = buf_out;
	struct hclge_dev *hdev = vport->back;
	struct hclge_desc desc;
	int ret;

	hclge_cmd_setup_basic_desc(&desc, HCLGE_OPC_DCQCN_GET_MSG_CNT, true);

	ret = hclge_cmd_send(&hdev->hw, &desc, 1);
	if (ret) {
		dev_err(&hdev->pdev->dev, "disable net lane failed %d\n", ret);
		return ret;
	}

	statistic_parm_out->dcqcn_rx_cnt = desc.data[0];
	statistic_parm_out->dcqcn_tx_cnt = desc.data[2];
	statistic_parm_out->dcqcn_db_cnt = desc.data[4];

	return 0;
}

struct drv_module_handle driv_module_cmd_handle[] = {
	{FW_VER, get_fw_ver},
	{DRIVER_VER, get_driver_ver},
	{CHECKSUM_CFG, hns3_test_chs_cfg},
	{TM_QUEUE_CFG, hns3_test_queue_cfg},
	{TM_QSET_CFG, hns3_test_qs_cfg},
	{TM_PRI_CFG, hns3_test_pri_cfg},
	{TM_PG_CFG, hns3_test_pg_cfg},
	{TM_PORT_CFG, hns3_test_port_cfg},
	{TM_ETS_CFG, hns3_test_ets_cfg},
	{DCB_MODE_CFG, hns3_test_dcb_cfg},
	{ETS_MODE_CFG, hns3_test_dcb_ets_cfg},
	{PFC_MODE_CFG, hns3_test_dcb_pfc_cfg},
	{MAC_LOOP_CFG, hns3_test_mac_loop_cfg},
	{DFX_INFO_CMD, hns3_test_get_dfx_info},
	{DFX_READ_CMD, hns3_test_read_dfx_info},
	{SEND_PKT, hns3_test_send_pkt},
	{RX_PRIV_BUFF_WL_CFG, hns3_test_rx_priv_buff_wl_cfg},
	{RX_COMMON_THRD_CFG, hns3_test_common_thrd_cfg},
	{RX_COMMON_WL_CFG, hns3_test_common_wl_cfg},
	{SHOW_RX_PRIV_WL, hns3_test_show_rx_priv_wl},
	{SHOW_RX_COMM_THRES, hns3_test_show_comm_thres},
	{QCN_EN_CFG, hns3_test_qcn_cfg},
	{RX_BUFF_CFG, hns3_test_rx_buff_cfg},
	{TX_BUFF_CFG, hns3_test_tx_buff_cfg},
	{RESET_CFG, hns3_nic_reset},
	{TIMEOUT_CFG, hns3_nic_timeout_cfg},
	{PROMISC_MODE_CFG, hns3_promisc_mode_cfg},
	{QINFO_CFG, hns3_test_qinfo_cfg},
	{PHY_REGISTER_CFG, hns3_test_phy_register_cfg},
	{MACTABLE_CFG, hns3_test_opt_mactbl},
	{CLEAN_STATS, hns3_test_clean_stats},
	{FD_CFG, hns3_test_fd_cfg},
	{RSS_GENERIC_CFG, hns3_test_rss_cfg},
	{REG_CFG, hns3_test_reg_cfg},
	{COM_REG_CFG, hns3_reg_cfg},
	{GRO_CFG, hns3_gro_age_handle},
	{M7_CMD_MODE_CFG, hns3_m7_cmd_handle},
	{QRES_CFG, hns3_test_qres_cfg},
	{STAT_CFG, hns3_stat_mode_cfg},
	{IRQ_CFG, hns3_irq_lli_cfg},
	{VLAN_UPMAPPING, hns3_test_upmapping_cfg},
	{LAMP_CFG, hns3_lamp_cfg},
	{EXTERN_INTERFACE_CFG, hns3_ext_interface_test},
	{XSFP_CFG, hns3_xsfp_cfg},
	{SHOW_PORT_INFO, hns3_get_port_info},
	{SHOW_HILINK_PARAM, hns3_get_hilink_param},
	{DCQCN_PARM_CFG, hns3_nic_dcqcn},
	{DCQCN_GET_MSG_CNT_CMD, hns3_dcqcn_get_msg_cnt}
};

static int send_to_driver(struct hns3_nic_priv *nic_dev,
			  struct msg_module *nt_msg,
			  void *buf_in, u32 in_size,
			  void *buf_out, u32 *out_size)
{
	u32 num_cmds = ARRAY_SIZE(driv_module_cmd_handle);
	enum driver_cmd_type cmd_type =
	    (enum driver_cmd_type)(nt_msg->msg_formate);
	driv_module fn;
	int err = 0;
	u32 index;

	for (index = 0; index < num_cmds; index++) {
		if (cmd_type == driv_module_cmd_handle[index].driv_cmd_name) {
			fn = driv_module_cmd_handle[index].driv_func;
			err = fn(nic_dev, buf_in, (u16)in_size, buf_out,
				 (u16 *)out_size);
			break;
		}
	}

	return err;
}

static long nictool_k_unlocked_ioctl(struct file *pfile, unsigned int cmd,
				     unsigned long arg)
{
	struct hns3_nic_priv *nic_dev = NULL;
	struct msg_module nt_msg;
	void *buf_out = NULL;
	void *buf_in = NULL;
	u32 out_size_expect;
	u32 out_size = 0;
	u32 in_size;
	int cmd_raw;
	int ret;

	memset(&nt_msg, 0, sizeof(nt_msg));

	if (copy_from_user(&nt_msg, (void *)arg, sizeof(nt_msg))) {
		pr_err("copy from user failed in unlocked_ioctl function\n");
		return -EFAULT;
	}

	cmd_raw = nt_msg.module;
	out_size_expect = nt_msg.len_info.out_buff_len;
	in_size = nt_msg.len_info.in_buff_len;
	ret = nictool_k_get_netdev_by_ifname(nt_msg.device_name, &nic_dev);
	if (ret) {
		pr_err("can not get the netdevice correctly\n");
		return -EINVAL;
	}

	if (nic_dev->ae_handle->flags & HNAE3_SUPPORT_VF) {
		pr_err("VF is not supported.\n");
		return -EINVAL;
	}

	ret = alloc_buff_in(&nt_msg, in_size, &buf_in);
	if (ret) {
		pr_err("alloc in buffer failed\n");
		return -EFAULT;
	}

	ret = alloc_buff_out(out_size_expect, &buf_out);
	if (ret) {
		pr_err("alloc out buffer failed\n");
		goto out_free_buf_in;
	}

	switch (cmd_raw) {
	case SEND_TO_DRIVER:
		ret = send_to_driver(nic_dev, &nt_msg, buf_in, in_size, buf_out,
				     &out_size);
		if (ret) {
			pr_err("send buffer to driver failed, ret = %d\n", ret);
			goto out_free_buf_out;
		}
		break;
	default:
		pr_err("module err!\n");
		ret = -EINVAL;
		goto out_free_buf_out;
	}

	ret = copy_buf_out_to_user(&nt_msg, out_size_expect, buf_out);
	if (ret)
		pr_err("copy buf to user failed\n");

out_free_buf_out:
	free_buff_out(buf_out);
out_free_buf_in:
	free_buff_in(buf_in);

	return (long)ret;
}

static int nictool_k_open(struct inode *pnode, struct file *pfile)
{
	return 0;
}

static ssize_t nictool_k_read(struct file *pfile, char __user *ubuf,
			      size_t size, loff_t *ppos)
{
	pr_info("%s read *ppos:%lld size = %d\n", __func__, *ppos, (int)size);
	return 0;
}

static ssize_t nictool_k_write(struct file *pfile, const char __user *ubuf,
			       size_t size, loff_t *ppos)
{
	pr_info("%s write *ppos:%lld size = %d\n", __func__, *ppos, (int)size);
	return 0;
}

int nictool_k_mmap(struct file *filp, struct vm_area_struct *vma)
{
	int ret;

	vma->vm_flags |= VM_IO;
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	ret = remap_pfn_range(vma, vma->vm_start, vma->vm_pgoff,
			      vma->vm_end - vma->vm_start, vma->vm_page_prot);
	if (ret)
		return -EIO;

	return 0;
}

static const struct file_operations fifo_operations = {
	.owner = THIS_MODULE,
	.open = nictool_k_open,
	.read = nictool_k_read,
	.write = nictool_k_write,
	.unlocked_ioctl = nictool_k_unlocked_ioctl,
	.mmap = nictool_k_mmap,
};

int if_nictool_exist(void)
{
	struct file *fp = NULL;
	int exist = 0;

	fp = filp_open("/dev/nic_dev", O_RDONLY, 0);
	if (IS_ERR(fp)) {
		exist = 0;
	} else {
		(void)filp_close(fp, NULL);
		exist = 1;
	}

	return exist;
}

int nictool_k_init(void)
{
	int ret;
	struct device *pdevice;

	if (g_nictool_init_flag) {
		g_nictool_ref_cnt++;
		return 0;
	}

	if (if_nictool_exist()) {
		pr_info("dev/nic_dev is existed!\n");
		return 0;
	}

	ret = alloc_chrdev_region(&g_dev_id, 0, 1, "nic_dev");
	if (ret < 0) {
		pr_err("alloc_chrdev_region fail, ret = %d.\n", ret);
		return ret;
	}

	g_nictool_class = class_create(THIS_MODULE, "nic_class");
	if (IS_ERR(g_nictool_class)) {
		pr_err("class create fail.\n");
		ret = -EFAULT;
		goto class_create_err;
	}

	cdev_init(&g_nictool_cdev, &fifo_operations);
	ret = cdev_add(&g_nictool_cdev, g_dev_id, 1);
	if (ret < 0) {
		pr_err("cdev_add fail, ret = %d.\n", ret);
		goto cdev_add_err;
	}

	pdevice = device_create(g_nictool_class, NULL, g_dev_id, NULL,
				"nic_dev");
	if (IS_ERR(pdevice)) {
		pr_err("device_create fail.\n");
		goto device_create_err;
	}

	g_nictool_init_flag = 1;
	g_nictool_ref_cnt = 1;
	pr_info("register nictool_dev to system, ok!\n");

	return 0;

device_create_err:
	cdev_del(&g_nictool_cdev);

cdev_add_err:
	class_destroy(g_nictool_class);

class_create_err:
	g_nictool_class = NULL;
	unregister_chrdev_region(g_dev_id, 1);

	return ret;
}

void nictool_k_uninit(void)
{
	if (g_nictool_init_flag) {
		if ((--g_nictool_ref_cnt))
			return;
	}

	if (!g_nictool_class || IS_ERR(g_nictool_class))
		return;

	cdev_del(&g_nictool_cdev);
	device_destroy(g_nictool_class, g_dev_id);
	class_destroy(g_nictool_class);
	g_nictool_class = NULL;
	unregister_chrdev_region(g_dev_id, 1);
	pr_info("unregister nictool_dev ok!\n");
}
