/*
 * Block size: B_YxB_X.
 * blockIdx.x determines case in batches of B_X*imgsPerThread, also color in batches of
 B_Y*colorsPerThread.
 *  In essence, blockIdx.x.x = 1..numImages/(B_X*imgsPerThread)
 *              blockIdx.x.y = 1..numImgColors/(B_Y*colorsPerThread)
 * blockIdx.y determines image pixel in target image.
 *
 * threadIdx.x determines case.
 * threadIdx.y determines color.
 *
 * hidActs:     (numFilters, numModulesY, numModulesX, numImages)
 * filters:     (numFilterColors, filterPixels, numFilters) if conv
 *              (numModulesY, numModulesX, numFilterColors, filterPixels, numFilters)
 otherwise
 * targets:     (numImageColors, imgSizeY, imgSizeX, numImages)
 *
 * Each block reconstructs one B_Y*colorsPerThread colors from 1 pixel from
 B_X*imgsPerThread cases.
 *
 * numImages must be divisible by B_X*imgsPerThread if checkCaseBounds is false.
 * numFiltersPerGroup must be divisible by filterCache.
 *
 * B_X * imgsPerThread must be divisible by 32.
 * numFilterColors must be divisible by B_Y*colorsPerThread.
 * B_X*B_Y must be divisible by 32.
 * filterCache must be divisible by B_X*B_Y/32
 * B_X*B_Y must be divisible by filterCache

 * This version loads 32 cases at a time, so it gets full coalescing on that load.
 * It only loads filterCache weights at a time, so those aren't fully coalesced
 (depending on size of filterCache).
 *
 * To be used when there are >= 16 color channels.
 */
template <
        int B_Y, int B_X, int imgsPerThread, int colorsPerThread, int filterCache,
        bool scale, bool checkCaseBounds, bool conv>
__global__ void conv_img_acts_manycolor(
        const float* hidActs, const float* filters, float* targets,
        const int numModulesY, const int numModulesX, const int numImages,
        const int numFilters, const int filterSize, const int imgSizeY,
        const int imgSizeX, const int paddingStart, const int moduleStride,
        const int numImgColors, const int numGroups, const float scaleTargets,
        const float scaleOutputs) {
    __shared__ float shFilters[colorsPerThread * B_Y][filterCache + 1];
    __shared__ float shHidActs[filterCache][B_X * imgsPerThread];
    fill_shared_mem<float>((float*)shFilters, sizeof(shFilters) / sizeof(float), 0);
    fill_shared_mem<float>((float*)shHidActs, sizeof(shHidActs) / sizeof(float), 0);
    __syncthreads();

    const int numImgBlocks = DIVUP(numImages, B_X * imgsPerThread);
    const int blockCaseIdx = (blockIdx.x % numImgBlocks) * B_X * imgsPerThread;

    const int imgColorIdx =
            (blockIdx.x / numImgBlocks) * B_Y * colorsPerThread;  // color idx globally
    const int numFilterColors = numImgColors / numGroups;
    const int blockGroupIdx = imgColorIdx / numFilterColors;
    const int filterColorIdx = imgColorIdx % numFilterColors;  // color idx within group
    const int numFiltersPerGroup = numFilters / numGroups;
    const int blockFilterIdx = blockGroupIdx * numFiltersPerGroup;

    const int blockPixelIdx = blockIdx.y;
    const int blockPixelIdxX = blockPixelIdx % imgSizeX;
    const int blockPixelIdxY = blockPixelIdx / imgSizeX;

    const int filterPixels = filterSize * filterSize;
    const int imgPixels = imgSizeY * imgSizeX;
    const int tidx = threadIdx.y * B_X + threadIdx.x;
    const int hidActLoadY = tidx / 32, hidActLoadX = tidx % 32;
    const int filtersLoadY = tidx / filterCache, filtersLoadX = tidx % filterCache;
    const int numModules = numModulesY * numModulesX;

    hidActs += blockCaseIdx + (blockFilterIdx + hidActLoadY) * numImages * numModules +
               hidActLoadX;
    filters += blockFilterIdx +
               (filterColorIdx + filtersLoadY) * filterPixels * numFilters +
               filtersLoadX;
    targets += (imgColorIdx + threadIdx.y) * imgPixels * numImages +
               blockPixelIdx * numImages + blockCaseIdx + threadIdx.x;

    float prod[colorsPerThread][imgsPerThread];
#pragma unroll
    for (int c = 0; c < colorsPerThread; c++) {
#pragma unroll
        for (int i = 0; i < imgsPerThread; i++) {
            prod[c][i] = 0;
        }
    }

    const int startY =
            blockPixelIdxY - paddingStart < filterSize
                    ? 0
                    : 1 + (blockPixelIdxY - paddingStart - filterSize) / moduleStride;
    const int endY =
            MIN(numModulesY, 1 + (blockPixelIdxY - paddingStart) / moduleStride);
    const int startX =
            blockPixelIdxX - paddingStart < filterSize
                    ? 0
                    : 1 + (blockPixelIdxX - paddingStart - filterSize) / moduleStride;
    const int endX =
            MIN(numModulesX, 1 + (blockPixelIdxX - paddingStart) / moduleStride);

    float* shFilterLoad = &shFilters[filtersLoadY][filtersLoadX];
    float* shHidActLoad = &shHidActs[hidActLoadY][hidActLoadX];

    for (int my = startY; my < endY; my++) {
        const int moduleTop = paddingStart + my * moduleStride;
        const int pxInFilterY = blockPixelIdxY - moduleTop;

        for (int mx = startX; mx < endX; mx++) {
            const int moduleIdx = my * numModulesX + mx;
            const int moduleLeft = paddingStart + mx * moduleStride;
            const int pxInFilterX = blockPixelIdxX - moduleLeft;

            const int pxIdxInFilter = pxInFilterY * filterSize + pxInFilterX;

            for (int f = 0; f < numFiltersPerGroup;
                 f += filterCache) {  // multiply with filterCache filters at a time
                const float* hLoad = &hidActs[(moduleIdx + f * numModules) * numImages];
#pragma unroll
                for (int i = 0; i < imgsPerThread * B_X; i += 32) {
                    if (!checkCaseBounds ||
                        blockCaseIdx + hidActLoadX + i < numImages) {
#pragma unroll
                        for (int j = 0; j < filterCache;
                             j += B_X * B_Y /
                                  32) {  // load filterCache rows of imgsPerThread*B_X
                                         // cols, 8 * 32 elements at a time.
                            shHidActLoad[j * B_X * imgsPerThread + i] =
                                    hLoad[j * numModules * numImages + i];
                        }
                    } else {
#pragma unroll
                        for (int j = 0; j < filterCache;
                             j += B_X * B_Y /
                                  32) {  // load filterCache rows of imgsPerThread*B_X
                                         // cols, 8 * 32 elements at a time.
                            shHidActLoad[j * B_X * imgsPerThread + i] = 0;
                        }
                    }
                }
                const float* fLoad =
                        conv ? &filters[pxIdxInFilter * numFilters + f]
                             : &filters
                                        [moduleIdx * numFilterColors * filterPixels *
                                                 numFilters +
                                         pxIdxInFilter * numFilters + f];
#pragma unroll
                for (int i = 0; i < colorsPerThread * B_Y;
                     i += B_X * B_Y / filterCache) {
                    if ((colorsPerThread * B_Y) % (B_X * B_Y / filterCache) == 0 ||
                        i + filtersLoadY < colorsPerThread * B_Y) {
                        shFilterLoad[i * (filterCache + 1)] =
                                fLoad[i * filterPixels * numFilters];
                    }
                }

                __syncthreads();
// Do some actual computation
#pragma unroll
                for (int i = 0; i < imgsPerThread; i++) {
#pragma unroll
                    for (int w = 0; w < filterCache; w++) {
#pragma unroll
                        for (int c = 0; c < colorsPerThread; c++) {
                            prod[c][i] += shFilters[c * B_Y + threadIdx.y][w] *
                                          shHidActs[w][threadIdx.x + i * B_X];
                        }
                    }
                }
                __syncthreads();
            }
        }
    }
    if (scale) {
#pragma unroll
        for (int i = 0; i < imgsPerThread; i++) {
            if (!checkCaseBounds || blockCaseIdx + threadIdx.x + i * B_X < numImages) {
#pragma unroll
                for (int c = 0; c < colorsPerThread; c++) {
                    targets[c * B_Y * imgPixels * numImages + i * B_X] =
                            scaleTargets *
                                    targets[c * B_Y * imgPixels * numImages + i * B_X] +
                            scaleOutputs * prod[c][i];
                }
            }
        }
    } else {
#pragma unroll
        for (int i = 0; i < imgsPerThread; i++) {
            if (!checkCaseBounds || blockCaseIdx + threadIdx.x + i * B_X < numImages) {
#pragma unroll
                for (int c = 0; c < colorsPerThread; c++) {
                    targets[c * B_Y * imgPixels * numImages + i * B_X] =
                            scaleOutputs * prod[c][i];
                }
            }
        }
    }
}
