---

layout: default

title: Serving responsive images in PhotoSwipe

h1_title: Responsive Images

description: The complete guide on how to serve responsive images in PhotoSwipe.

addjs: true

canonical_url: http://photoswipe.com/documentation/responsive-images.html

buildtool: true

markdownpage: true

---

PhotoSwipe 不支持 `<picture>` 或 `srcset`，因为它需要定义的图像尺寸并使用惰性加载。但是，由于图像是动态加载的，所以很容易切换数据源，即使在不支持 `srcset` 的旧浏览器中也是如此。

让我们假设你只有“中等”图像和“原始”（“大”）图像。首先，你需要在幻灯片对象中存储图像的路径和大小，例如如下所示:

```javascript
var items = [

	// Slide 1
	{
		mediumImage: {
			src: 'path/to/medium-image-1.jpg',
			w:800,
			h:600
		},
		originalImage: {
			src: 'path/to/large-image-1.jpg',
			w: 1400,
			h: 1050
		}
	},

	// Slide 2
	// {
	//     mediumImage: {
	//         src: 'path/to/medium-image-2.jpg',
	//         ...
	//     
	// ...
		
];
```

然后:

```javascript
// initialise as usual
var gallery = new PhotoSwipe( pswpElement, PhotoSwipeUI_Default, items, options);

// create variable that will store real size of viewport
var realViewportWidth,
	useLargeImages = false,
	firstResize = true,
	imageSrcWillChange;

// beforeResize event fires each time size of gallery viewport updates
gallery.listen('beforeResize', function() {
	// gallery.viewportSize.x - width of PhotoSwipe viewport
	// gallery.viewportSize.y - height of PhotoSwipe viewport
	// window.devicePixelRatio - ratio between physical pixels and device independent pixels (Number)
	//							1 (regular display), 2 (@2x, retina) ...
	

	// calculate real pixels when size changes
	realViewportWidth = gallery.viewportSize.x * window.devicePixelRatio;
	
	// Code below is needed if you want image to switch dynamically on window.resize
	
	// Find out if current images need to be changed
	if(useLargeImages && realViewportWidth < 1000) {
		useLargeImages = false;
		imageSrcWillChange = true;
	} else if(!useLargeImages && realViewportWidth >= 1000) {
		useLargeImages = true;
		imageSrcWillChange = true;
	}

	// Invalidate items only when source is changed and when it's not the first update
	if(imageSrcWillChange && !firstResize) {
		// invalidateCurrItems sets a flag on slides that are in DOM,
		// which will force update of content (image) on window.resize.
		gallery.invalidateCurrItems();
	}

	if(firstResize) {
		firstResize = false;
	}

	imageSrcWillChange = false;

});


// gettingData event fires each time PhotoSwipe retrieves image source & size
gallery.listen('gettingData', function(index, item) {

	// Set image source & size based on real viewport width
	if( useLargeImages ) {
		item.src = item.originalImage.src;
		item.w = item.originalImage.w;
		item.h = item.originalImage.h;
	} else {
		item.src = item.mediumImage.src;
		item.w = item.mediumImage.w;
		item.h = item.mediumImage.h;
	}

	// It doesn't really matter what will you do here, 
	// as long as item.src, item.w and item.h have valid values.
	// 
	// Just avoid http requests in this listener, as it fires quite often

});


// Note that init() method is called after gettingData event is bound
gallery.init();

```

- 你不必使用与上面完全相同的幻灯片对象结构（使用 `mediumImage` 和 `largeImage` 对象）。例如，你可以将图像的大小直接存储在图像文件名（`/path/to/large-image-600x500）中。然后在 `gettingData` 事件中解析大小。只有 `item.src`、`item.w` 和 `item.h` 属性被 PhotoSwipe 读取，并且只有在 `gettingData` 事件被触发之后。
- 图像越大，动画看起来就越不流畅。
- 尽量避免仅基于 DeviceXelRatio 或仅基于视口大小提供图像，始终将两者结合起来。
- 在打开 PhotoSwipe 的缩略图上随意使用 `srcset`。

知道如何改进这本指南吗？[建议编辑一下！](https://github.com/dimsemenov/PhotoSwipe/blob/master/website/documentation/responsive-images.md)

