# 设置 CA 并生成 TLS 证书

在这个实验室中，你将使用 Cloudflare 的 PKI 工具包 [PKI 基础设施](https://en.wikipedia.org/wiki/Public_key_infrastructure) 提供 [PKI 基础设施](https://en.wikipedia.org/wiki/Public_key_infrastructure)，然后使用它引导证书颁发机构，并为以下组件生成 TLS 证书:ETCD、Kube-Apiserver、Kube-Controller-Manager、Kube-Scheduler、Kubelet 和 Kube-Proxy。

## 证书颁发机构

在本节中，你将提供一个证书授权机构，该机构可用于生成额外的 TLS 证书。

生成 CA 配置文件、证书和私钥:

```
{

cat > ca-config.json <<EOF
{
  "signing": {
    "default": {
      "expiry": "8760h"
    },
    "profiles": {
      "kubernetes": {
        "usages": ["signing", "key encipherment", "server auth", "client auth"],
        "expiry": "8760h"
      }
    }
  }
}
EOF

cat > ca-csr.json <<EOF
{
  "CN": "Kubernetes",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "Kubernetes",
      "OU": "CA",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert -initca ca-csr.json | cfssljson -bare ca

}
```

结果:

```
ca-key.pem
ca.pem
```

## 客户机和服务器证书

在本节中，你将为每个 Kubernetes组件生成客户机和服务器证书，并为 Kubernetes`admin` 用户生成客户机证书。

### 管理客户端证书

生成 `admin` 客户端证书和私钥:

```
{

cat > admin-csr.json <<EOF
{
  "CN": "admin",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "system:masters",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -profile=kubernetes \
  admin-csr.json | cfssljson -bare admin

}
```

结果:

```
admin-key.pem
admin.pem
```

### Kubelet 客户端证书

Kubernetes 使用一个名为 node authorizer 的 [特殊用途授权模式](https://kubernetes.io/docs/admin/authorization/node/)，专门授权由 [Kubelets](https://kubernetes.io/docs/concepts/overview/components/#kubelet) 发出的 API 请求。为了获得节点授权人的授权，Kubelets 必须使用一个凭证，该凭证将它们标识为位于 `system:nodes` 组中，用户名为 `system:node:<nodeName>`。在本节中，你将为每个 Kubernetes 工作节点创建一个满足节点授权人要求的证书。

为每个 Kubernetes 工作者节点生成一个证书和私钥:

```
for instance in worker-0 worker-1 worker-2; do
cat > ${instance}-csr.json <<EOF
{
  "CN": "system:node:${instance}",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "system:nodes",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

EXTERNAL_IP=$(gcloud compute instances describe ${instance} \
  --format 'value(networkInterfaces[0].accessConfigs[0].natIP)')

INTERNAL_IP=$(gcloud compute instances describe ${instance} \
  --format 'value(networkInterfaces[0].networkIP)')

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -hostname=${instance},${EXTERNAL_IP},${INTERNAL_IP} \
  -profile=kubernetes \
  ${instance}-csr.json | cfssljson -bare ${instance}
done
```

结果:

```
worker-0-key.pem
worker-0.pem
worker-1-key.pem
worker-1.pem
worker-2-key.pem
worker-2.pem
```

### Controller Manager 客户端证书

生成 `kube-controller-manager` 客户端证书和私钥:

```
{

cat > kube-controller-manager-csr.json <<EOF
{
  "CN": "system:kube-controller-manager",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "system:kube-controller-manager",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -profile=kubernetes \
  kube-controller-manager-csr.json | cfssljson -bare kube-controller-manager

}
```

结果:

```
kube-controller-manager-key.pem
kube-controller-manager.pem
```

### Kube 代理客户端证书

生成 `kube-proxy` 客户端证书和私钥:

```
{

cat > kube-proxy-csr.json <<EOF
{
  "CN": "system:kube-proxy",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "system:node-proxier",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -profile=kubernetes \
  kube-proxy-csr.json | cfssljson -bare kube-proxy

}
```

结果:

```
kube-proxy-key.pem
kube-proxy.pem
```

### 调度程序客户端证书

生成 `kube-scheduler` 客户端证书和私钥:

```
{

cat > kube-scheduler-csr.json <<EOF
{
  "CN": "system:kube-scheduler",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "system:kube-scheduler",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -profile=kubernetes \
  kube-scheduler-csr.json | cfssljson -bare kube-scheduler

}
```

结果:

```
kube-scheduler-key.pem
kube-scheduler.pem
```

### Kubernetes API 服务器证书

`kubernetes-the-hard-way` 静态 IP 地址将包含在 Kubernetes API 服务器证书的主题备选名称列表中。这将确保证书可以由远程客户端验证。

生成 Kubernetes API 服务器证书和私钥:

```
{

KUBERNETES_PUBLIC_ADDRESS=$(gcloud compute addresses describe kubernetes-the-hard-way \
  --region $(gcloud config get-value compute/region) \
  --format 'value(address)')

KUBERNETES_HOSTNAMES=kubernetes,kubernetes.default,kubernetes.default.svc,kubernetes.default.svc.cluster,kubernetes.svc.cluster.local

cat > kubernetes-csr.json <<EOF
{
  "CN": "kubernetes",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "Kubernetes",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -hostname=10.32.0.1,10.240.0.10,10.240.0.11,10.240.0.12,${KUBERNETES_PUBLIC_ADDRESS},127.0.0.1,${KUBERNETES_HOSTNAMES} \
  -profile=kubernetes \
  kubernetes-csr.json | cfssljson -bare kubernetes

}
```

> Kubernetes API 服务器将自动分配 `kubernetes` 内部 DNS 名称，该名称将从 [控制平面引导](08-bootstrapping-kubernetes-controllers.md#configure-the-kubernetes-api-server) 实验室期间为内部集群服务保留的地址范围（`10.32.0.0`）链接到第一个 IP 地址（`10.32.0.1`）。

结果:

```
kubernetes-key.pem
kubernetes.pem
```

## 服务帐户密钥对

Kubernetes Controller Manager 利用一个密钥对来生成和签名服务帐户令牌，如 [管理服务账户](https://kubernetes.io/docs/admin/service-accounts-admin/)文档中所述。

生成 `service-account` 证书和私钥:

```
{

cat > service-account-csr.json <<EOF
{
  "CN": "service-accounts",
  "key": {
    "algo": "rsa",
    "size": 2048
  },
  "names": [
    {
      "C": "US",
      "L": "Portland",
      "O": "Kubernetes",
      "OU": "Kubernetes The Hard Way",
      "ST": "Oregon"
    }
  ]
}
EOF

cfssl gencert \
  -ca=ca.pem \
  -ca-key=ca-key.pem \
  -config=ca-config.json \
  -profile=kubernetes \
  service-account-csr.json | cfssljson -bare service-account

}
```

结果:

```
service-account-key.pem
service-account.pem
```

## 分发客户机和服务器证书

将适当的证书和私钥复制到每个 worker 实例:

```
for instance in worker-0 worker-1 worker-2; do
  gcloud compute scp ca.pem ${instance}-key.pem ${instance}.pem ${instance}:~/
done
```

将适当的证书和私钥复制到每个控制器实例:

```
for instance in controller-0 controller-1 controller-2; do
  gcloud compute scp ca.pem ca-key.pem kubernetes-key.pem kubernetes.pem \
    service-account-key.pem service-account.pem ${instance}:~/
done
```

> `kube-proxy`、`kube-controller-manager`、`kube-scheduler` 和 `kubelet` 客户端证书将用于在下一个实验室中生成客户端身份验证配置文件。

下一条:[生成 Kubernetes 配置文件以进行身份验证](05-kubernetes-configuration-files.md)

