import unittest

from parsec.state import BasicState

from skill_tree.exercises.marked_math import processor
import skill_tree.exercises.markdown as mk


def math_processor(context):
    """ math(str)->str
对文本内容预处理，将公式标记为前端可展示的 html。
    """
    md = context
    new_md = []
    math_ctx = {
        "enter": False,
        "chars": []
    }

    count = len(md)
    i = 0
    while i < count:
        c = md[i]
        if c == '$':
            if math_ctx['enter']:
                j = 0
                chars = math_ctx['chars']
                length = len(chars)
                while j < length:
                    cc = chars[j]
                    if cc == '_':
                        next_c = chars[j + 1]
                        if next_c == '{':
                            subs = []
                            cursor = 2
                            next_c = chars[j + cursor]
                            while next_c != '}':
                                subs.append(next_c)
                                cursor += 1
                                next_c = chars[j + cursor]

                            sub = ''.join(subs)
                            new_md.append(f'<sub>{sub}</sub>')
                            j += cursor
                        else:
                            new_md.append(f'<sub>{next_c}</sub>')
                            j += 1
                    elif cc == '^':
                        next_c = chars[j + 1]
                        if next_c == '{':
                            subs = []
                            cursor = 2
                            next_c = chars[j + cursor]
                            while next_c != '}':
                                subs.append(next_c)
                                cursor += 1
                                next_c = chars[j + cursor]

                            sub = ''.join(subs)
                            new_md.append(f'<sup>{sub}</sup>')
                            j += cursor
                        else:
                            new_md.append(f'<sup>{next_c}</sup>')
                            j += 1
                    else:
                        new_md.append(cc)
                    j += 1

                math_ctx['enter'] = False
                math_ctx['chars'] = []
            else:
                math_ctx['enter'] = True
                math_ctx['chars'] = []
        else:
            if math_ctx['enter']:
                math_ctx['chars'].append(c)
            else:
                new_md.append(c)
        i += 1
    return "".join(new_md)

data = """
# Hello World

以下 `Hello World` 程序中，能够正确输出内容的是：

## 答案

```java
package app;

public class App {
    public static void main(String[] args){
        System.out.println("Hello World");
    }
}
```

## 选项

### B

```java
package app;

public class App {
    public int main(){
        System.out.printf("Hello World");
        return 0;
    }
}
```

### C

```java
package app;

public class App {
    public static void main(String[] args){
        println("Hello World");
    }
}
```

### D

```java
package app;
import stdout

public class App {
    public int main(){
        print("Hello World\n");
        return 0;
    }
}

```

"""


class MarkdownTestCase(unittest.TestCase):
    def test_basic(self):
        state = BasicState(data.strip())
        doc = mk.parse(state)
        self.assertEqual(doc.title, "Hello World")
        self.assertEqual(len(doc.options), 3)
        self.assertEqual(doc.description[0].source.strip(), """以下 `Hello World` 程序中，能够正确输出内容的是：""")
        self.assertEqual(doc.answer[0].language, "java")
        self.assertEqual(doc.answer[0].source, """package app;

public class App {
    public static void main(String[] args){
        System.out.println("Hello World");
    }
}""")
        self.assertEqual(doc.options[0].paras[0].language, "java")
        self.assertEqual(doc.options[0].paras[0].source, """package app;

public class App {
    public int main(){
        System.out.printf("Hello World");
        return 0;
    }
}""")
        self.assertEqual(doc.options[1].paras[0].language, "java")
        self.assertEqual(doc.options[1].paras[0].source, """package app;

public class App {
    public static void main(String[] args){
        println("Hello World");
    }
}""")
        self.assertEqual(doc.options[2].paras[0].language, "java")
        self.assertEqual(doc.options[2].paras[0].source, """package app;
import stdout

public class App {
    public int main(){
        print("Hello World\n");
        return 0;
    }
}
""")


class MathTestCase(unittest.TestCase):
    def test_parse(self):
        data = "$e^{pi}$"
        result = processor(data)
        self.assertEqual(result, math_processor(data))

    def test_pack(self):
        data = "ploy is $e^{pi}$ in plain text"
        result = processor(data)
        self.assertEqual(result, math_processor(data))

    def test_simple(self):
        data = "ploy is $x_0$ in plain text"
        result = processor(data)
        self.assertEqual(result, math_processor(data))

    def test_left(self):
        data = "$x_0$ at start of plain text"
        result = processor(data)
        self.assertEqual(result, math_processor(data))

    def test_right(self):
        data = "ploy is $x_0$"
        result = processor(data)
        self.assertEqual(result, math_processor(data))

spark = """
# Neo4j的Apache Spark连接器

Apache Spark 的 Neo4j 连接器旨在使图与 Spark 的集成变得容易。有两种有效的使用连接器的方法：

- **作为数据源**：在 Spark 中将任何节点或关系集作为 DataFrame 读取
- **作为接收器**：将任何 DataFrame 作为节点或关系的集合写入 Neo4j，或者；使用 Cypher 语句将 DataFrame 中的记录处理为您选择的图模式。

由于连接器基于新的 Spark 数据源 API，因此适用于 Python 和 R 等语言的其他 Spark 解释器也可以使用。

此连接器适用于 Neo4j 3.5 和 Neo4j 的整个 4+ 系列，无论是作为单个实例运行、在因果集群模式下运行，还是作为 Neo4j Aura 中的托管服务运行。连接器不依赖于企业功能，因此也可以与 Neo4j 社区一起使用，并具有适当的版本号。

此连接器当前支持 Spark 2.4.5+ with Scala 2.11 和 Scala 2.12 以及 Spark 3.0+ with Scala 2.12。根据 Spark 和 Scala 版本的组合，您将需要不同的 JAR。JAR 以以下形式命名`neo4j-connector-apache-spark_${scala.version}_${spark.version}_${connector.version}`

问题：

下列说法中，不正确的一项是？

## 答案

Spark连接器在开源社区版上无法使用

## 选项

### A

Spark 连接器可用于从Spark中读取或写入数据

### B

Spark 连接器使用Java驱动程序来连接底层Neo4j图数据库

### C

Spark擅长处理DataFrame，Neo4j擅长处理图数据，使用连接器可以发挥两个的优势

### D

Spark连接器在开源社区版上无法使用
"""

learn = """
# 代码块

请点击[代码块](https://codechina.csdn.net/courses/register/2/8){target="_blank"} 并完成所有题目，完成学习请关闭Issue。

通过对[代码块](https://codechina.csdn.net/courses/register/2/8){target="_blank"}的学习，以下哪个代码块语法不对？

## 答案

```bash
行内代码块 ```hello```
```

## 选项

### A

```bash
行内代码块 `hello`
```

### B

````bash
```diff
var foo = 'bar';
+ var x = 200;
* var x = 100;
```
````

### C

````bash
```python
def fibo(max):
    n, a, b = 0, 0, 1
    while n < max:
        yield b
        a, b = b, a + b
        n = n + 1  
for n in fibo(10):
    print (n)
```
````
"""

class MarkdownBlockTestCase(unittest.TestCase):
    def test_parse(self):
        state = BasicState(learn.strip())
        doc = mk.parse(state)
        self.assertEqual("```diff\nvar foo = 'bar';\n+ var x = 200;\n* var x = 100;\n```",
                         doc.options[1].paras[0].source)


shell = """
# SHELL 的变量

下面说法中不正确的是

## 答案

下面的代码扩展了系统变量 $PATH

```
export $PATH=$PATH:$HOME/bin
```

## 选项

### A

Shell 通常使用 `$abc` 的形式引用变量

### B

下面的代码扩展了系统变量 $PATH

```
export PATH=$PATH:$HOME/bin
```

### C

通过下列命令可以查看 PATH 

```
echo $PATH
```
"""

class DollarTestCase(unittest.TestCase):
    def test_parse(self):
        state = BasicState(shell.strip())
        doc = mk.parse(state)
        self.assertEqual("Shell 通常使用 `$abc` 的形式引用变量",
                         doc.options[0].paras[0].source)
        self.assertEqual("""下面的代码扩展了系统变量 $PATH""",
                         doc.options[1].paras[0].source)
        self.assertEqual("""export PATH=$PATH:$HOME/bin""",
                         doc.options[1].paras[1].source)
        self.assertEqual(3, len(doc.options))