# opencv实现目标检测功能

早在 2017 年 8 月，OpenCV 3.3 正式发布，带来了高度改进的“深度神经网络”（dnn）模块。 该模块支持多种深度学习框架，包括 Caffe、TensorFlow 和 Torch/PyTorch。这次我们使用Opencv深度学习的功能实现目标检测的功能，模型选用MobileNetSSD_deploy.caffemodel。

我已经实现了模型加载和预测的代码，但是如何将结果取出来并绘制到图片上呢？如下图：

![output](https://gitee.com/wanghao1090220084/cloud-image/raw/master/output.jpg)

# 框架代码

```Python
import numpy as np
import cv2
if __name__=="__main__":
	image_name = '11.jpg'
	prototxt = 'MobileNetSSD_deploy.prototxt.txt'
	model_path = 'MobileNetSSD_deploy.caffemodel'
	confidence_ta = 0.2
	# 初始化MobileNet SSD训练的类标签列表
	# 检测，然后为每个类生成一组边界框颜色
	CLASSES = ["background", "aeroplane", "bicycle", "bird", "boat",
			   "bottle", "bus", "car", "cat", "chair", "cow", "diningtable",
			   "dog", "horse", "motorbike", "person", "pottedplant", "sheep",
			   "sofa", "train", "tvmonitor"]
	COLORS = np.random.uniform(0, 255, size=(len(CLASSES), 3))
	# load our serialized model from disk
	print("[INFO] loading model...")
	net = cv2.dnn.readNetFromCaffe(prototxt, model_path)
	# 加载输入图像并为图像构造一个输入blob
	# 将大小调整为固定的300x300像素。
	# （注意：SSD模型的输入是300x300像素）
	image = cv2.imread(image_name)
	(h, w) = image.shape[:2]
	blob = cv2.dnn.blobFromImage(cv2.resize(image, (300, 300)), 0.007843,
								 (300, 300), 127.5)
	# 通过网络传递blob并获得检测结果和
	# 预测
	print("[INFO] computing object detections...")
	net.setInput(blob)
	detections = net.forward()
	# 循环检测结果
	
	# show the output image
	cv2.imshow("Output", image)
	cv2.imwrite("output.jpg", image)
	cv2.waitKey(0)


```

# 答案：

```Python
for i in np.arange(0, detections.shape[2]):
		# 提取与数据相关的置信度（即概率）
		# 预测
		confidence = detections[0, 0, i, 2]
		# 通过确保“置信度”来过滤掉弱检测
		# 大于最小置信度
		if confidence > confidence_ta:
			# 从`detections`中提取类标签的索引，
			# 然后计算物体边界框的 (x, y) 坐标
			idx = int(detections[0, 0, i, 1])
			box = detections[0, 0, i, 3:7] * np.array([w, h, w, h])
			(startX, startY, endX, endY) = box.astype("int")
			# 显示预测
			label = "{}: {:.2f}%".format(CLASSES[idx], confidence * 100)
			print("[INFO] {}".format(label))
			cv2.rectangle(image, (startX, startY), (endX, endY),
						  COLORS[idx], 2)
			y = startY - 15 if startY - 15 > 15 else startY + 15
			cv2.putText(image, label, (startX, y),
						cv2.FONT_HERSHEY_SIMPLEX, 0.5, COLORS[idx], 2)
```

# 选项

## 宽和高颠倒

```Python
for i in np.arange(0, detections.shape[2]):
		# 提取与数据相关的置信度（即概率）
		# 预测
		confidence = detections[0, 0, i, 2]
		# 通过确保“置信度”来过滤掉弱检测
		# 大于最小置信度
		if confidence > confidence_ta:
			# 从`detections`中提取类标签的索引，
			# 然后计算物体边界框的 (x, y) 坐标
			idx = int(detections[0, 0, i, 1])
			box = detections[0, 0, i, 3:7] * np.array([h, w, h, w])
			(startX, startY, endX, endY) = box.astype("int")
			# 显示预测
			label = "{}: {:.2f}%".format(CLASSES[idx], confidence * 100)
			print("[INFO] {}".format(label))
			cv2.rectangle(image, (startX, startY), (endX, endY),
						  COLORS[idx], 2)
			y = startY - 15 if startY - 15 > 15 else startY + 15
			cv2.putText(image, label, (startX, y),
						cv2.FONT_HERSHEY_SIMPLEX, 0.5, COLORS[idx], 2)
```

## box框顺序错误

```
for i in np.arange(0, detections.shape[2]):
		# 提取与数据相关的置信度（即概率）
		# 预测
		confidence = detections[0, 0, i, 2]
		# 通过确保“置信度”来过滤掉弱检测
		# 大于最小置信度
		if confidence > confidence_ta:
			# 从`detections`中提取类标签的索引，
			# 然后计算物体边界框的 (x, y) 坐标
			idx = int(detections[0, 0, i, 1])
			box = detections[0, 0, i, 3:7] * np.array([w, h, w, h])
			(startX, endX, startY, endY) = box.astype("int")
			# 显示预测
			label = "{}: {:.2f}%".format(CLASSES[idx], confidence * 100)
			print("[INFO] {}".format(label))
			cv2.rectangle(image, (startX, startY), (endX, endY),
						  COLORS[idx], 2)
			y = startY - 15 if startY - 15 > 15 else startY + 15
			cv2.putText(image, label, (startX, y),
						cv2.FONT_HERSHEY_SIMPLEX, 0.5, COLORS[idx], 2)
```

## 置信度设置成大于

```
for i in np.arange(0, detections.shape[2]):
		# 提取与数据相关的置信度（即概率）
		# 预测
		confidence = detections[0, 0, i, 2]
		# 通过确保“置信度”来过滤掉弱检测
		# 大于最小置信度
		if confidence < confidence_ta:
			# 从`detections`中提取类标签的索引，
			# 然后计算物体边界框的 (x, y) 坐标
			idx = int(detections[0, 0, i, 1])
			box = detections[0, 0, i, 3:7] * np.array([w, h, w, h])
			(startX, startY, endX, endY) = box.astype("int")
			# 显示预测
			label = "{}: {:.2f}%".format(CLASSES[idx], confidence * 100)
			print("[INFO] {}".format(label))
			cv2.rectangle(image, (startX, startY), (endX, endY),
						  COLORS[idx], 2)
			y = startY - 15 if startY - 15 > 15 else startY + 15
			cv2.putText(image, label, (startX, y),
						cv2.FONT_HERSHEY_SIMPLEX, 0.5, COLORS[idx], 2)
```

