# 自定义运行时异常（RuntimeException）

现有如下的海伦公式计算函数，根据三角形的三条边长`a`、`b`和`c`（均确保大于0）计算三角形的面积。当三角形本身非法时返回负数，否则返回面积，如下所示

```java
public class TestMain {
    private static float getArea(float a, float b, float c) {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            return -1;  // 非法三角形
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);  // 海伦公式
        }
    }

    public static void main(String[] args) {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

请改写上述代码，对于三角形非法的情况，**仅在运行抛出不可查异常（Unchecked Exception）**而不是返回`-1`。

> 不可查异常（Unchecked Exception）指在运行时发生的异常。这些也被称为运行时异常（Runtime Exception）。其中包括程序逻辑错误或API使用不当等，例如ArrayIndexOutOfBoundsException、IllegalStateException与NullPointerException等，此类异常将在编译时忽略，仅在运行时处理。

因此，下列修改方式正确的是：

## 答案

```java
public class TestMain {
    public static class InvalidTriangleException extends RuntimeException {
        public InvalidTriangleException(float a, float b, float c) {
            super(String.format("Invalid triangle - (%.3f, %.3f, %.3f).", a, b, c));
        }
    }

    private static float getArea(float a, float b, float c) {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            throw new InvalidTriangleException(a, b, c);
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);
        }
    }

    public static void main(String[] args) {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

## 选项

### A

```java
public class TestMain {
    public class InvalidTriangleException extends RuntimeException {
        public InvalidTriangleException(float a, float b, float c) {
            super(String.format("Invalid triangle - (%.3f, %.3f, %.3f).", a, b, c));
        }
    }

    private float getArea(float a, float b, float c) {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            throw new InvalidTriangleException(a, b, c);
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);
        }
    }

    public static void main(String[] args) {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

### B

```java
public class TestMain {
    public static class InvalidTriangleException extends Exception {
        public InvalidTriangleException(float a, float b, float c) {
            super(String.format("Invalid triangle - (%.3f, %.3f, %.3f).", a, b, c));
        }
    }

    private static float getArea(float a, float b, float c) {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            throw new InvalidTriangleException(a, b, c);
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);
        }
    }

    public static void main(String[] args) {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

### C

```java
public class TestMain {
    public static class InvalidTriangleException extends Exception {
        public InvalidTriangleException(float a, float b, float c) {
            super(String.format("Invalid triangle - (%.3f, %.3f, %.3f).", a, b, c));
        }
    }

    private static float getArea(float a, float b, float c) throws InvalidTriangleException {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            throw new InvalidTriangleException(a, b, c);
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);
        }
    }

    public static void main(String[] args) throws InvalidTriangleException {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

### D

```java
public class TestMain {
    public static class InvalidTriangleException extends Error {
        public InvalidTriangleException(float a, float b, float c) {
            super(String.format("Invalid triangle - (%.3f, %.3f, %.3f).", a, b, c));
        }
    }

    private static float getArea(float a, float b, float c) {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            throw new InvalidTriangleException(a, b, c);
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);
        }
    }

    public static void main(String[] args) {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

### E

```java
public class TestMain {
    public static class InvalidTriangleException extends RuntimeException {
        public InvalidTriangleException(float a, float b, float c) {
            super(String.format("Invalid triangle - (%.3f, %.3f, %.3f).", a, b, c));
        }
    }

    private static float getArea(float a, float b, float c) throws InvalidTriangleException {
        if ((a + b < c) || (a + c < b) || (b + c < a)) {
            throw new InvalidTriangleException(a, b, c);
        } else {
            float p = (a + b + c) / 2;
            return (float) Math.sqrt((p - a) * (p - b) * (p - c) * p);
        }
    }

    public static void main(String[] args) throws InvalidTriangleException {
        System.out.println(TestMain.getArea(3, 4, 5));
    }
}
```

