# 资源对象管理

现有如下的程序，可以从文件`myfile.txt`中读取文本内容，将行号以标准的格式输出到标准输出流，对于可能存在的异常情况（如文件不存在、无权限等）需要进行妥善处理，且需要**保证当文件读取完毕后文件对象被关闭**，如下所示

```java
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

public class TestMain {
    public static void main(String[] args) throws IOException {
        String filePath = "myfile.txt";
        InputStream fileStream = null;
        try {
            fileStream = new FileInputStream(filePath);
            Scanner scanner = new Scanner(fileStream);
            int lineno = 0;
            while (scanner.hasNextLine()) {
                lineno += 1;
                String line = scanner.nextLine();
                System.out.printf("%3d >>> %s%n", lineno, line);
            }
            System.out.println("(EOF)");
        } catch (IOException e) {
            System.err.printf("Error occurred when reading '%s'.%n", filePath);
            System.exit(-1);
        } finally {
            if (fileStream != null) {
                fileStream.close();
            }
        }
    }
}

```

请改写上述代码，使用`try_with_resource`的方法简化程序，并依然**确保文件对象在任意情况下均被关闭**。

因此，下列修改方式正确的是：

## 答案

```java
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

public class TestMain {
    public static void main(String[] args) {
        String filePath = "myfile.txt";
        try (
                InputStream fileStream = new FileInputStream(filePath);
                Scanner scanner = new Scanner(fileStream)
        ) {
            int lineno = 0;
            while (scanner.hasNextLine()) {
                lineno += 1;
                String line = scanner.nextLine();
                System.out.printf("%3d >>> %s%n", lineno, line);
            }
            System.out.println("(EOF)");
        } catch (IOException e) {
            System.err.printf("Error occurred when reading '%s'.%n", filePath);
            System.exit(-1);
        }
    }
}
```

## 选项

### A

```java
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

public class TestMain {
    public static void main(String[] args) {
        String filePath = "myfile.txt";
        try {
            InputStream fileStream = new FileInputStream(filePath);
            Scanner scanner = new Scanner(fileStream);
        } {
            int lineno = 0;
            while (scanner.hasNextLine()) {
                lineno += 1;
                String line = scanner.nextLine();
                System.out.printf("%3d >>> %s%n", lineno, line);
            }
            System.out.println("(EOF)");
        } catch (IOException e) {
            System.err.printf("Error occurred when reading '%s'.%n", filePath);
            System.exit(-1);
        }
    }
}
```

### B

```java
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

public class TestMain {
    public static void main(String[] args) {
        String filePath = "myfile.txt";
        try (fileStream, scanner = new FileInputStream(filePath), new Scanner(fileStream)) {
            int lineno = 0;
            while (scanner.hasNextLine()) {
                lineno += 1;
                String line = scanner.nextLine();
                System.out.printf("%3d >>> %s%n", lineno, line);
            }
            System.out.println("(EOF)");
        } catch (IOException e) {
            System.err.printf("Error occurred when reading '%s'.%n", filePath);
            System.exit(-1);
        }
    }
}
```

### C

```java
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

public class TestMain {
    public static void main(String[] args) {
        String filePath = "myfile.txt";
        try (InputStream fileStream = new FileInputStream(filePath)) {
            Scanner scanner = new Scanner(fileStream);
            int lineno = 0;
            while (scanner.hasNextLine()) {
                lineno += 1;
                String line = scanner.nextLine();
                System.out.printf("%3d >>> %s%n", lineno, line);
            }
            System.out.println("(EOF)");
        } catch (IOException e) {
            System.err.printf("Error occurred when reading '%s'.%n", filePath);
            System.exit(-1);
        }
    }
}
```

### D

```java
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Scanner;

public class TestMain {
    public static void main(String[] args) {
        String filePath = "myfile.txt";
        try (
                InputStream fileStream = new FileInputStream(filePath);
                Scanner scanner = new Scanner(fileStream);
                int lineno = 0
        ) {
            while (scanner.hasNextLine()) {
                lineno += 1;
                String line = scanner.nextLine();
                System.out.printf("%3d >>> %s%n", lineno, line);
            }
            System.out.println("(EOF)");
        } catch (IOException e) {
            System.err.printf("Error occurred when reading '%s'.%n", filePath);
            System.exit(-1);
        }
    }
}
```

