# 自定义可查异常（Checked Exception）

现有如下的学生考试成绩查询程序，会根据给定的字符串格式`studentID`进行学生分数查询，如果该学生存在，则返回其分数（确保为非负整数）；如果该学生不存在，则返回`-1`，如下所示

```java
public class TestMain {
    private static boolean containsStudent(String studentID) {
        // Return true when the given `studentID` represents an existing student.
    }

    private static int queryScore(String studentID) {
        // Get and then return score of the student
        // whose ID is `StudentID` from the database.
    }

    private static int getScoreByStudentID(String studentID) {
        if (containsStudent(studentID)) {
            return queryScore(studentID);
        } else {
            return -1;
        }
    }

    public static void main(String[] args) {
        String studentID = "ID20211224";

        int score = getScoreByStudentID(studentID);
        if (score >= 0) {
            System.out.printf("Student ID : %s%n", studentID);
            System.out.printf("Score : %d%n", score);
        } else {
            System.out.printf("Student ID %s not found!%n", studentID);
        }
    }
}

```

请改写上述代码，对于学生不存在的情况，**使用可查异常（Checked Exception）对潜在的异常情况进行妥善处理**而不是返回`-1`。

> 可查异常（Checked Exception）是编译器在编译时会进行检查，也称为编译时异常（Compile Time Exception）。 这些异常往往具备可预见性，且不能简单地被忽略，编程者应当对其进行妥善处理。

因此，下列修改方式正确的是：

## 答案

```java
public class TestMain {
    private static boolean containsStudent(String studentID) {
        // Return true when the given `studentID` represents an existing student.
    }

    private static int queryScore(String studentID) {
        // Get and then return score of the student
        // whose ID is `StudentID` from the database.
    }

    public static class StudentNotExistException extends Exception {
        public StudentNotExistException(String studentID) {
            super(String.format("Student ID %s not found.", studentID));
        }
    }

    private static int getScoreByStudentID(String studentID) throws StudentNotExistException {
        if (containsStudent(studentID)) {
            return queryScore(studentID);
        } else {
            throw new StudentNotExistException(studentID);
        }
    }

    public static void main(String[] args) {
        String studentID = "ID20211224";

        try {
            int score = getScoreByStudentID(studentID);
            System.out.printf("Student ID : %s%n", studentID);
            System.out.printf("Score : %d%n", score);
        } catch (StudentNotExistException err) {
            System.out.printf("Student ID %s not found!%n", studentID);
        }
    }
}
```

## 选项

### A

```java
public class TestMain {
    private static boolean containsStudent(String studentID) {
        // Return true when the given `studentID` represents an existing student.
    }

    private static int queryScore(String studentID) {
        // Get and then return score of the student
        // whose ID is `StudentID` from the database.
    }

    public static class StudentNotExistException extends RuntimeException {
        public StudentNotExistException(String studentID) {
            super(String.format("Student ID %s not found.", studentID));
        }
    }

    private static int getScoreByStudentID(String studentID) throws StudentNotExistException {
        if (containsStudent(studentID)) {
            return queryScore(studentID);
        } else {
            throw new StudentNotExistException(studentID);
        }
    }

    public static void main(String[] args) {
        String studentID = "ID20211224";

        try {
            int score = getScoreByStudentID(studentID);
            System.out.printf("Student ID : %s%n", studentID);
            System.out.printf("Score : %d%n", score);
        } catch (StudentNotExistException err) {
            System.out.printf("Student ID %s not found!%n", studentID);
        }
    }
}
```

### B

```java
public class TestMain {
    private static boolean containsStudent(String studentID) {
        // Return true when the given `studentID` represents an existing student.
    }

    private static int queryScore(String studentID) {
        // Get and then return score of the student
        // whose ID is `StudentID` from the database.
    }

    public static class StudentNotExistException extends RuntimeException {
        public StudentNotExistException(String studentID) {
            super(String.format("Student ID %s not found.", studentID));
        }
    }

    private static int getScoreByStudentID(String studentID) {
        if (containsStudent(studentID)) {
            return queryScore(studentID);
        } else {
            throw new StudentNotExistException(studentID);
        }
    }

    public static void main(String[] args) {
        String studentID = "ID20211224";

        int score = getScoreByStudentID(studentID);
        System.out.printf("Student ID : %s%n", studentID);
        System.out.printf("Score : %d%n", score);
    }
}
```

### C

```java
public class TestMain {
    private static boolean containsStudent(String studentID) {
        // Return true when the given `studentID` represents an existing student.
    }

    private static int queryScore(String studentID) {
        // Get and then return score of the student
        // whose ID is `StudentID` from the database.
    }

    public static class StudentNotExistException extends Error {
        public StudentNotExistException(String studentID) {
            super(String.format("Student ID %s not found.", studentID));
        }
    }

    private static int getScoreByStudentID(String studentID) throws StudentNotExistException {
        if (containsStudent(studentID)) {
            return queryScore(studentID);
        } else {
            throw new StudentNotExistException(studentID);
        }
    }

    public static void main(String[] args) {
        String studentID = "ID20211224";

        try {
            int score = getScoreByStudentID(studentID);
            System.out.printf("Student ID : %s%n", studentID);
            System.out.printf("Score : %d%n", score);
        } catch (StudentNotExistException err) {
            System.out.printf("Student ID %s not found!%n", studentID);
        }
    }
}
```

### D

```java
public class TestMain {
    private static boolean containsStudent(String studentID) {
        // Return true when the given `studentID` represents an existing student.
    }

    private static int queryScore(String studentID) {
        // Get and then return score of the student
        // whose ID is `StudentID` from the database.
    }

    public static class StudentNotExistException extends Exception {
        public StudentNotExistException(String studentID) {
            super(String.format("Student ID %s not found.", studentID));
        }
    }

    private static int getScoreByStudentID(String studentID) throws StudentNotExistException {
        if (containsStudent(studentID)) {
            return queryScore(studentID);
        } else {
            throw new StudentNotExistException(studentID);
        }
    }

    public static void main(String[] args) {
        String studentID = "ID20211224";

        try {
            int score = getScoreByStudentID(studentID);
            System.out.printf("Student ID : %s%n", studentID);
            System.out.printf("Score : %d%n", score);
        } except (StudentNotExistException err) {
            System.out.printf("Student ID %s not found!%n", studentID);
        }
    }
}
```

