package cn.bugstack.domain.strategy.service.armory;

import cn.bugstack.domain.strategy.model.entity.StrategyAwardEntity;
import cn.bugstack.domain.strategy.model.entity.StrategyEntity;
import cn.bugstack.domain.strategy.model.entity.StrategyRuleEntity;
import cn.bugstack.domain.strategy.repository.IStrategyRepository;
import cn.bugstack.types.enums.ResponseCode;
import cn.bugstack.types.exception.AppException;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.security.SecureRandom;
import java.util.*;

/**
 * @description:
 * @author: hdr
 * @PACKAGE_NAME: cn.bugstack.domain.strategy.service.armory
 * @DATE: 2024/4/5
 */
@Slf4j
@Service
public class StrategyArmory implements IStrategyArmory, IStrategyDispatch {

    @Autowired
    private IStrategyRepository repository;

    public static void main(String[] args) {
        StrategyArmory strategyArmory = new StrategyArmory();
        strategyArmory.assembleLotteryStrategy(100001L);
    }

    @Override
    public boolean assembleLotteryStrategy(Long strategy) {

        // 1. 查询策略配置
        List<StrategyAwardEntity> awardEntityList = repository.queryStrategyAwardList(strategy);
        assembleLotteryStrategy(String.valueOf(strategy),awardEntityList);

        // 2. 权重策略陪住 - 使用于 rule_weight 权重规则配置
        StrategyEntity strategyEntity = repository.queryStrategyEntityByStrategyId(strategy);

        if (StringUtils.isEmpty(strategyEntity.getRuleWeight())) return true;
        String ruleWeight = strategyEntity.getRuleWeight();
                // 2.2 查询策略规则
        StrategyRuleEntity strategyRuleEntity = repository.queryStrategyRule(strategy, ruleWeight);
        if (null == strategyRuleEntity) {
            throw new AppException(ResponseCode.STRATEGY_RULE_WEIGHT_IS_NULL.getCode(), ResponseCode.STRATEGY_RULE_WEIGHT_IS_NULL.getInfo());
        }
        Map<String, List<Integer>> ruleWeightValueMap = strategyRuleEntity.getRuleWeightValues();
        Set<String> keys = ruleWeightValueMap.keySet();
        // 2.3 将获取到的所有 规则写入缓存
        for (String key : keys) {
            List<Integer> ruleWeightValues = ruleWeightValueMap.get(key);
            ArrayList<StrategyAwardEntity> strategyAwardEntitiesClone = new ArrayList<>(awardEntityList);
            strategyAwardEntitiesClone.removeIf(entity -> !ruleWeightValues.contains(entity.getAwardId()));
            assembleLotteryStrategy(String.valueOf(strategy).concat("_").concat(key), strategyAwardEntitiesClone);
        }
        return true;

    }

    private void assembleLotteryStrategy(String key,List<StrategyAwardEntity> awardEntityList) {
        // 1. 获取最小概率值
        BigDecimal minAwardRate = awardEntityList.stream()
                .map(StrategyAwardEntity::getAwardRate)
                .min(BigDecimal::compareTo)
                .orElse(BigDecimal.ZERO);

        // 2. 获取概率值总和
        BigDecimal totalAwardRate = awardEntityList.stream()
                .map(StrategyAwardEntity::getAwardRate)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        // 3. 用 1% 0.001获取概率范围，百分位、千分位、万分位
        BigDecimal rateRange = totalAwardRate.divide(minAwardRate, 0, RoundingMode.CEILING);


        ArrayList<Object> strategyAwardSearchRateTables = new ArrayList(rateRange.intValue());
        for (StrategyAwardEntity strategyAwardEntity : awardEntityList) {
            Integer awardId = strategyAwardEntity.getAwardId();
            BigDecimal awardRate = strategyAwardEntity.getAwardRate();

            // 4. 计算出每个概率值需要存储到查找表的数量，循环填充
            for (int i = 0; i < rateRange.multiply(awardRate).setScale(0, RoundingMode.CEILING).intValue(); i++) {
                strategyAwardSearchRateTables.add(awardId);
            }

            // 5. 乱序
            Collections.shuffle(strategyAwardSearchRateTables);

            // 6. 组装抽奖表
            HashMap<Integer, Integer> shuffleStrategyAwardSearchRateTables = new HashMap<>();
            for (int i = 0; i < strategyAwardSearchRateTables.size(); i++) {
                shuffleStrategyAwardSearchRateTables.put(i, (Integer) strategyAwardSearchRateTables.get(i));
            }

            // 7. 存储到 redis中
            repository.storeStrategyAwardSearchRateTables(key,shuffleStrategyAwardSearchRateTables.size(), shuffleStrategyAwardSearchRateTables);
        }
    }

    @Override
    public Integer getRandomAwardId(Long strategyId) {
        int rateRange = repository.getRandomAwardId(strategyId);
        return repository.getStrategyAwardAssemble(String.valueOf(strategyId),new SecureRandom().nextInt(rateRange));
    }

    @Override
    public Integer getRandomAwardId(Long strategyId, String ruleWeightValue) {
        String key = String.valueOf(strategyId).concat("_").concat(ruleWeightValue);
        // 分布式部署下，不一定为当前应用做的策略装配。也就是值不一定会保存到本应用，而是分布式应用，所以需要从 Redis 中获取。
        int rateRange = repository.getRateRange(key);
        // 通过生成的随机值，获取概率值奖品查找表的结果
        return repository.getStrategyAwardAssemble(key, new SecureRandom().nextInt(rateRange));
    }
}
