package cn.bugstack.infrastructure.persistent.repository;

import cn.bugstack.domain.strategy.model.entity.StrategyAwardEntity;
import cn.bugstack.domain.strategy.model.entity.StrategyEntity;
import cn.bugstack.domain.strategy.model.entity.StrategyRuleEntity;
import cn.bugstack.domain.strategy.model.valobj.StrategyAwardRuleModelVO;
import cn.bugstack.domain.strategy.repository.IStrategyRepository;
import cn.bugstack.infrastructure.persistent.dao.IStrategyAwardDao;
import cn.bugstack.infrastructure.persistent.dao.IStrategyDao;
import cn.bugstack.infrastructure.persistent.dao.IStrategyRuleDao;
import cn.bugstack.infrastructure.persistent.po.Strategy;
import cn.bugstack.infrastructure.persistent.po.StrategyAward;
import cn.bugstack.infrastructure.persistent.po.StrategyRule;
import cn.bugstack.infrastructure.persistent.redis.IRedisService;
import cn.bugstack.types.common.Constants;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Repository;

import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @description: 策略仓储实现
 * @author: hdr
 * @PACKAGE_NAME: cn.bugstack.infrastructure.persistent.repository
 * @DATE: 2024/4/7
 */
@Repository
@Slf4j
public class StrategyRepository implements IStrategyRepository {

    @Autowired
    private IStrategyAwardDao strategyAwardDao;

    @Autowired
    private IRedisService redisService;

    @Autowired
    private IStrategyDao strategyDao;

    @Autowired
    private IStrategyRuleDao strategyRuleDao;


    @Override
    public List<StrategyAwardEntity> queryStrategyAwardList(Long strategy) {
        // 1. 查询redis中是否有保存策略数据有的话直接返回
        String cacheKey = Constants.RedisKey.STRATEGY_AWARD_KEY + strategy;
        List<StrategyAwardEntity> strategyAwardEntities = redisService.getValue(cacheKey);

        if (strategyAwardEntities != null && strategyAwardEntities.size() != 0) {
//            List<StrategyAwardEntity> strategyAwardEntities = JSON.parseArray(json.toString(), StrategyAwardEntity.class);
            return strategyAwardEntities;
        }

        // 2. 没有保存策略数据则查询数据库
        List<StrategyAward> strategyAwards = strategyAwardDao.queryStrategyAwardListByStrategyId(strategy);
        List<StrategyAwardEntity> strategyAwardEntityList = strategyAwards.stream().map(item -> {
            StrategyAwardEntity strategyAwardEntity = new StrategyAwardEntity();
            strategyAwardEntity.setStrategyId(item.getStrategyId());
            strategyAwardEntity.setAwardId(item.getAwardId());
            strategyAwardEntity.setAwardCount(item.getAwardCount());
            strategyAwardEntity.setAwardRate(item.getAwardRate());
            strategyAwardEntity.setAwardCountSurplus(item.getAwardCountSurplus());
            return strategyAwardEntity;
        }).collect(Collectors.toList());

        // 2.1 保存到redis中

        redisService.setValue(cacheKey, strategyAwardEntityList);
        return strategyAwardEntityList;
    }

    @Override
    public void storeStrategyAwardSearchRateTables(String key, Integer rateRange, HashMap<Integer, Integer> shuffleStrategyAwardSearchRateTables) {
        // 1. 存储概率值
        redisService.setValue(Constants.RedisKey.STRATEGY_RATE_RANGE_KEY + key, rateRange.intValue());

        // 2. 存储概率查找表
        Map<Integer, Integer> cacheRateTable = redisService.getMap(Constants.RedisKey.STRATEGY_RATE_TABLE_KEY + key);
        cacheRateTable.putAll(shuffleStrategyAwardSearchRateTables);
    }

    @Override
    public int getRandomAwardId(Long strategyId) {
        return getRateRange(String.valueOf(strategyId));
    }

    @Override
    public Integer getStrategyAwardAssemble(String key, int nextInt) {
        return redisService.getFromMap(Constants.RedisKey.STRATEGY_RATE_TABLE_KEY + key,nextInt);
    }

    @Override
    public StrategyEntity queryStrategyEntityByStrategyId(Long strategyId) {
        // 1. 优先从缓存获取
        String cacheKey = Constants.RedisKey.STRATEGY_KEY + strategyId;

        StrategyEntity strategyEntity = new StrategyEntity();
        StrategyEntity maps = redisService.getValue(cacheKey);
        if (null != maps){
//            strategyEntity = StrategyEntity.builder()
//                    .strategyId(Long.parseLong(maps.get("strategyId").toString()))
//                    .strategyDesc((String) maps.get("strategyDesc"))
//                    .ruleModels((String) maps.get("ruleModels"))
//                    .build();
            return maps;
        }
        // 2. 没有数据从数据库获取
        Strategy strategy = strategyDao.queryStrategyByStrategyId(strategyId);
        strategyEntity.setStrategyId(strategy.getStrategyId());
        if (StringUtils.isNotBlank(strategy.getRuleModels())) {
            strategyEntity.setRuleModels(strategy.getRuleModels());
        }
        strategyEntity.setStrategyDesc(strategy.getStrategyDesc());
//                = StrategyEntity.builder()
//                .strategyId(strategy.getStrategyId())
//                .strategyDesc(strategy.getStrategyDesc())
//                .ruleModels(strategy.getRuleModels())
//                .build();


        // 3. 写入缓存
        log.info("strategyEntity：{}",strategyEntity);
        redisService.setValue(cacheKey,strategyEntity);
        return strategyEntity;
    }

    @Override
    public StrategyRuleEntity queryStrategyRule(Long strategyId, String ruleModel) {
        StrategyRule strategyRuleReq = new StrategyRule();
        strategyRuleReq.setStrategyId(strategyId);
        strategyRuleReq.setRuleModel(ruleModel);
        StrategyRule strategyRuleRes = strategyRuleDao.queryStrategyRule(strategyRuleReq);
        return StrategyRuleEntity.builder()
                .strategyId(strategyRuleRes.getStrategyId())
                .awardId(strategyRuleRes.getAwardId())
                .ruleType(strategyRuleRes.getRuleType())
                .ruleModel(strategyRuleRes.getRuleModel())
                .ruleValue(strategyRuleRes.getRuleValue())
                .ruleDesc(strategyRuleRes.getRuleDesc())
                .build();
    }

    @Override
    public int getRateRange(String key) {
        return redisService.getValue(Constants.RedisKey.STRATEGY_RATE_RANGE_KEY + key);
    }

    @Override
    public String queryStrategyRuleValue(Long strategyId, String ruleModel) {
        return queryStrategyRuleValue(strategyId, null, ruleModel);
    }

    @Override
    public String queryStrategyRuleValue(Long strategyId, Integer awardId, String ruleModel) {
        StrategyRule strategyRule = new StrategyRule();
        strategyRule.setStrategyId(strategyId);
        strategyRule.setRuleModel(ruleModel);
        strategyRule.setAwardId(awardId);
        return strategyRuleDao.queryStrategyRuleValue(strategyRule);
    }

    @Override
    public StrategyAwardRuleModelVO queryStrategyAwardRuleModelVO(Long strategyId, Integer awardId) {
        StrategyAward strategyAward = new StrategyAward();
        strategyAward.setStrategyId(strategyId);
        strategyAward.setAwardId(awardId);
        String ruleModels = strategyAwardDao.queryStrategyAwardRuleModels(strategyAward);

        return StrategyAwardRuleModelVO.builder().ruleModels(ruleModels).build();
    }

}
