# 第10天：策略模式设计

## 📚 今日目标

1. 理解策略模式的概念
2. 理解为什么用策略模式
3. 设计路由策略接口
4. 为后续实现做准备

---

## 🎯 知识点1：什么是策略模式？

### 生活中的例子

**支付方式**：
- 支付宝支付
- 微信支付
- 银行卡支付

**策略模式**：
- 定义一系列算法
- 把它们封装起来
- 使它们可以互换

### 策略模式的结构

```
策略接口（Strategy）
    ↑
    |
策略实现1（ConcreteStrategy1）
策略实现2（ConcreteStrategy2）
策略实现3（ConcreteStrategy3）
```

---

## 🎯 知识点2：为什么用策略模式？

### 问题场景

**不同的路由算法**：
- 哈希路由：hash(userId) % dbCount
- 取模路由：userId % dbCount
- 范围路由：根据范围判断

**问题**：如果不用策略模式
```java
// 硬编码，不灵活
if (algorithm.equals("hash")) {
    // 哈希算法
} else if (algorithm.equals("mod")) {
    // 取模算法
}
```

**解决方案**：策略模式
```java
// 灵活，易扩展
IDBRouterStrategy strategy = getStrategy(algorithm);
strategy.doRouter(userId);
```

---

## 🛠️ 实践任务：设计路由策略接口

### 步骤1：创建IDBRouterStrategy接口

在 `src/main/java/cn/bugstack/middleware/db/router/strategy/` 目录下创建 `IDBRouterStrategy.java`：

```java
package cn.bugstack.middleware.db.router.strategy;

/**
 * 数据库路由策略接口
 * 
 * 定义路由算法的规范
 * 
 * @author 小傅哥
 */
public interface IDBRouterStrategy {

    /**
     * 执行路由
     * 
     * @param dbKeyAttr 路由键的值
     */
    void doRouter(String dbKeyAttr);

    /**
     * 设置数据库索引
     * 
     * @param dbIdx 数据库索引
     */
    void setDBKey(int dbIdx);

    /**
     * 设置表索引
     * 
     * @param tbIdx 表索引
     */
    void setTBKey(int tbIdx);

    /**
     * 获取数据库数量
     * 
     * @return 数据库数量
     */
    int dbCount();

    /**
     * 获取表数量
     * 
     * @return 表数量
     */
    int tbCount();

    /**
     * 清理路由信息
     */
    void clear();
}
```

### 代码解释

1. **doRouter方法**：
   - 执行路由逻辑
   - 根据路由键计算库和表索引
   - 设置到ThreadLocal

2. **setDBKey/setTBKey**：
   - 设置库和表索引
   - 用于手动指定路由

3. **dbCount/tbCount**：
   - 获取库和表数量
   - 用于路由计算

4. **clear方法**：
   - 清理路由信息
   - 避免内存泄漏

---

## 🎓 知识点拓展

### 拓展1：策略模式 vs 工厂模式

**策略模式**：
- 关注算法的选择
- 运行时选择算法

**工厂模式**：
- 关注对象的创建
- 创建时选择对象

**结合使用**：
```java
// 工厂创建策略
IDBRouterStrategy strategy = StrategyFactory.create("hash");
// 策略执行算法
strategy.doRouter(userId);
```

### 拓展2：策略模式的优缺点

**优点**：
- ✅ 算法可以自由切换
- ✅ 避免使用多重条件判断
- ✅ 扩展性良好

**缺点**：
- ❌ 客户端必须知道所有策略类
- ❌ 策略类数量增多

### 拓展3：其他设计模式

**模板方法模式**：
- 定义算法骨架
- 子类实现具体步骤

**观察者模式**：
- 定义一对多依赖
- 一个对象改变，所有依赖者收到通知

---

## ✅ 今日检查清单

- [ ] 理解了策略模式的概念
- [ ] 理解了为什么用策略模式
- [ ] 设计了路由策略接口
- [ ] 理解了策略模式的优缺点
- [ ] 完成了拓展阅读

---

## 🎯 明日预告

明天我们将开始核心实现：
- 实现配置属性类DBRouterConfig
- 实现工具类StringUtils
- 完善项目基础

---

## 💡 思考题

1. 为什么用策略模式而不是if-else？
2. 策略模式和工厂模式的区别？
3. 如何扩展新的路由策略？

---

## 📚 参考资源

- [设计模式：策略模式](https://refactoring.guru/design-patterns/strategy)
- [Java设计模式](https://www.runoob.com/design-pattern/strategy-pattern.html)
- [策略模式实战](https://www.baeldung.com/java-strategy-pattern)
