/* Copyright (c) 2016 PaddlePaddle Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. */

#include "paddle/fluid/operators/sequence_ops/sequence_slice_op.h"

#include <memory>

namespace paddle {
namespace operators {

class SequenceSliceOp : public framework::OperatorWithKernel {
 public:
  using framework::OperatorWithKernel::OperatorWithKernel;

  void InferShape(framework::InferShapeContext* ctx) const override {
    OP_INOUT_CHECK(ctx->HasInput("X"), "Input", "X", "SequenceSlice");
    OP_INOUT_CHECK(ctx->HasInput("Offset"), "Input", "Offset", "SequenceSlice");
    OP_INOUT_CHECK(ctx->HasInput("Length"), "Input", "Length", "SequenceSlice");
    OP_INOUT_CHECK(ctx->HasOutput("Out"), "Output", "Out", "SequenceSlice");
    auto input_dims = ctx->GetInputDim("X");

    auto offset_dim = ctx->GetInputDim("Offset");
    auto length_dim = ctx->GetInputDim("Length");

    PADDLE_ENFORCE_EQ(
        offset_dim.size(),
        2UL,
        platform::errors::InvalidArgument(
            "Input Offset dimension error. SequenceSlice operator only support "
            "one level sequence now, the dimension of input Offset must be 2, "
            "but received dimension is %d.",
            offset_dim.size()));
    PADDLE_ENFORCE_EQ(
        length_dim.size(),
        2UL,
        platform::errors::InvalidArgument(
            "Input Length dimension error. SequenceSlice operator only support "
            "one level sequence now, the dimension of input Length must be 2, "
            "but received dimension is %d.",
            offset_dim.size()));

    // Initialize the output's dims to maximum,
    // and re-set to real dims by the value of Offset and Length at kernel
    ctx->SetOutputDim("Out", input_dims);
  }

 protected:
  phi::KernelKey GetExpectedKernelType(
      const framework::ExecutionContext& ctx) const override {
    return phi::KernelKey(OperatorWithKernel::IndicateVarDataType(ctx, "X"),
                          ctx.GetPlace());
  }
};

class SequenceSliceGradOp : public framework::OperatorWithKernel {
 public:
  using framework::OperatorWithKernel::OperatorWithKernel;

  void InferShape(framework::InferShapeContext* ctx) const override {
    OP_INOUT_CHECK(ctx->HasInput(framework::GradVarName("Out")),
                   "Input",
                   framework::GradVarName("Out"),
                   "SequenceSliceGrad");
    OP_INOUT_CHECK(ctx->HasOutputs(framework::GradVarName("X")),
                   "Output",
                   framework::GradVarName("X"),
                   "SequenceSliceGrad");
    ctx->SetOutputsDim(framework::GradVarName("X"), ctx->GetInputsDim("X"));
  }

 protected:
  phi::KernelKey GetExpectedKernelType(
      const framework::ExecutionContext& ctx) const override {
    return phi::KernelKey(OperatorWithKernel::IndicateVarDataType(
                              ctx, framework::GradVarName("Out")),
                          ctx.GetPlace());
  }
};

class SequenceSliceOpMaker : public framework::OpProtoAndCheckerMaker {
 public:
  void Make() override {
    AddInput("X",
             "(LoDTensor), "
             "the input of SequenceSliceOp.");
    AddInput("Offset",
             "(Tensor), "
             "a vector<int> to describe the offset of every input sequence for "
             "sub sequence item.");
    AddInput("Length",
             "(Tensor), "
             "a vector<int> to describe the length of every input sequence for "
             "sub sequence item.");
    AddOutput("Out", "(LoDTensor), the output of SequenceSliceOp.");
    AddComment(R"DOC(
Sequence slice operator

The operator crops a subsequence from given sequence with given start offset and subsequence length.
It only supports sequence (LoD Tensor with level number is 1).
- Case:
    X = [[a1, a2;
        b1, b2;
        c1, c2]
       [d1, d2;
        e1, e2]]
    LoD(X) = {{0, 3, 5}}; Dims(X) = (5, 2)
    Offset = [[0], [1]]; Length = [[2], [1]]

    Out = [[a1, a2;
            b1, b2]
            [e1, e2]]
    LoD(Out) = {{0, 2, 3}}; Dims(Out) = (3, 2)
NOTE: The first dimension size of input, the size of offset and Length, should be equal. The offset start from 0.
    )DOC");
  }
};

template <typename T>
class SequenceSliceGradOpMaker : public framework::SingleGradOpMaker<T> {
 public:
  using framework::SingleGradOpMaker<T>::SingleGradOpMaker;

 protected:
  void Apply(GradOpPtr<T> op) const override {
    op->SetType("sequence_slice_grad");
    op->SetInput("X", this->Input("X"));
    op->SetInput("Offset", this->Input("Offset"));
    op->SetInput("Length", this->Input("Length"));
    op->SetInput(framework::GradVarName("Out"), this->OutputGrad("Out"));
    op->SetOutput(framework::GradVarName("X"), this->InputGrad("X"));
    op->SetAttrMap(this->Attrs());
  }
};

DECLARE_NO_NEED_BUFFER_VARS_INFERER(SequenceSliceGradNoNeedBufferVarsInferer,
                                    "X");

}  // namespace operators
}  // namespace paddle

namespace ops = paddle::operators;
REGISTER_OPERATOR(sequence_slice,
                  ops::SequenceSliceOp,
                  ops::SequenceSliceOpMaker,
                  ops::SequenceSliceGradOpMaker<paddle::framework::OpDesc>,
                  ops::SequenceSliceGradOpMaker<paddle::imperative::OpBase>);
REGISTER_OPERATOR(sequence_slice_grad,
                  ops::SequenceSliceGradOp,
                  ops::SequenceSliceGradNoNeedBufferVarsInferer);
REGISTER_OP_CPU_KERNEL(sequence_slice,
                       ops::SequenceSliceOpKernel<phi::CPUContext, float>,
                       ops::SequenceSliceOpKernel<phi::CPUContext, double>,
                       ops::SequenceSliceOpKernel<phi::CPUContext, int>,
                       ops::SequenceSliceOpKernel<phi::CPUContext, int64_t>);
REGISTER_OP_CPU_KERNEL(
    sequence_slice_grad,
    ops::SequenceSliceGradOpKernel<phi::CPUContext, float>,
    ops::SequenceSliceGradOpKernel<phi::CPUContext, double>,
    ops::SequenceSliceGradOpKernel<phi::CPUContext, int>,
    ops::SequenceSliceGradOpKernel<phi::CPUContext, int64_t>);
